/* abc - The AspectBench Compiler
 * Copyright (C) 2004 Polyglot project group
 * Copyright (C) 2004 Laurie Hendren
 * Copyright (C) 2004 Pavel Avgustinov
 *
 * This compiler is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This compiler is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this compiler, in the file LESSER-GPL;
 * if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Java 1.4 scanner for JFlex.
 * Based on JLS, 2ed, Chapter 3.
 * Adapted for abc Pavel Avgustinov <pavel.avgustinov@magd.ox.ac.uk>, August 2004.
 */

package abc.aspectj.parse;
%%
import java_cup.runtime.Symbol;
import polyglot.lex.*;
import polyglot.util.Position;
import polyglot.util.ErrorQueue;
import polyglot.util.ErrorInfo;
import java.util.HashMap;
import java.util.Stack;
import polyglot.ext.jl.parse.*;
import java.math.BigInteger;
%%

%layout aspectj

%option public "%public"
//TWEAK
%option class "%class Lexer_c"
%option implements "%implements polyglot.lex.Lexer"
%option type "%type Token"
%option function "%function nextToken"
%option unicode "%unicode"
%option pack "%pack"
%option line "%line"
%option column "%column"

%{

    StringBuffer sb = new StringBuffer();
    String file;
    ErrorQueue eq;

    public Lexer_c(java.io.InputStream in, String file, ErrorQueue eq) {
        this(new java.io.BufferedReader(new java.io.InputStreamReader(in)),
             file, eq);
    }
    
    public Lexer_c(java.io.Reader reader, String file, ErrorQueue eq) {
        this(new EscapedUnicodeReader(reader));
        this.file = file;
        this.eq = eq;
    }

    public String file() {
        return file;
    }

    private Position pos() {
        return new Position(file, yyline+1, yycolumn, yyline+1, yycolumn+yytext().length());
    }
    private Position pos(int len) {
        return new Position(file, yyline+1, yycolumn-len-1, yyline+1, yycolumn+1);
    }

    private Token key(int symbol) {
        return new Keyword(pos(), yytext(), symbol);
    }

    private Token op(int symbol) {
        return new Operator(pos(), yytext(), symbol);
    }

    private Token id() {
        return new Identifier(pos(), yytext(), sym.IDENTIFIER);
    }

    /* ---- added for id patterns, needed in Pointcuts  --- */
    private Token id_pattern() {
	//System.out.println("ID pattern: " + yytext());
        return new Identifier(pos(), yytext(), sym.IDENTIFIERPATTERN);
    }

    private Token int_lit(String s, int radix) {
	    BigInteger x = new BigInteger(s, radix);
	    boolean boundary = (radix == 10 && s.equals("2147483648"));
	    int bits = (radix == 10 ? 31 : 32);
	    if (x.bitLength() > bits && !boundary) { 
			eq.enqueue(ErrorInfo.LEXICAL_ERROR, "Integer literal \"" +
			   yytext() + "\" out of range.", pos());
	    }
	    return new IntegerLiteral(pos(), x.intValue(), 
	    		boundary? sym.INTEGER_LITERAL_BD : sym.INTEGER_LITERAL);
    }

    private Token long_lit(String s, int radix) {
	    BigInteger x = new BigInteger(s, radix);
	    boolean boundary = (radix == 10 && s.equals("9223372036854775808"));
        int bits = (radix == 10 ? 63 : 64);
	    if (x.bitLength() > bits && !boundary) {
			eq.enqueue(ErrorInfo.LEXICAL_ERROR, "Long literal \"" +
			   yytext() + "\" out of range.", pos());
			return new LongLiteral(pos(), x.longValue(), sym.LONG_LITERAL); // null;
	    }
	    return new LongLiteral(pos(), x.longValue(), 
	    		boundary? sym.LONG_LITERAL_BD : sym.LONG_LITERAL);
    }

    private Token float_lit(String s) {
        try {
            Float x = Float.valueOf(s);
		    boolean zero = true;
		    for (int i = 0; i < s.length(); i++) {
				if ('1' <= s.charAt(i) && s.charAt(i) <= '9') {
				    zero = false;
				    break;
				}
				else if(s.charAt(i) == 'e' || s.charAt(i) == 'E') {
					break; // 0e19 is still 0
				}
		    }
		    if (x.isInfinite() || x.isNaN() || (x.floatValue() == 0 && ! zero)) {
				eq.enqueue(ErrorInfo.LEXICAL_ERROR,
				   "Illegal float literal \"" + yytext() + "\"", pos());
				return new FloatLiteral(pos(), 0, sym.FLOAT_LITERAL); // null;
		    }
            return new FloatLiteral(pos(), x.floatValue(), sym.FLOAT_LITERAL);
        }
        catch (NumberFormatException e) {
            eq.enqueue(ErrorInfo.LEXICAL_ERROR,
                       "Illegal float literal \"" + yytext() + "\"", pos());
            return new FloatLiteral(pos(), 0, sym.FLOAT_LITERAL); // null;
        }
    }

    private Token double_lit(String s) {
        try {
            Double x = Double.valueOf(s);
		    boolean zero = true;
		    for (int i = 0; i < s.length(); i++) {
				if ('1' <= s.charAt(i) && s.charAt(i) <= '9') {
				    zero = false;
				    break;
				}
				else if(s.charAt(i) == 'e' || s.charAt(i) == 'E') {
					break; // 0e19 is still 0
				}
		    }
		    if (x.isInfinite() || x.isNaN() || (x.doubleValue() == 0 && ! zero)) {
				eq.enqueue(ErrorInfo.LEXICAL_ERROR,
				   "Illegal double literal \"" + yytext() + "\"", pos());
				return new DoubleLiteral(pos(), 0, sym.DOUBLE_LITERAL); // null;
		    }
            return new DoubleLiteral(pos(), x.doubleValue(), sym.DOUBLE_LITERAL);
        }
        catch (NumberFormatException e) {
            eq.enqueue(ErrorInfo.LEXICAL_ERROR,
                       "Illegal double literal \"" + yytext() + "\"", pos());
            return new DoubleLiteral(pos(), 0, sym.DOUBLE_LITERAL); // null;
        }
    }

    private Token char_lit(String s) {
        if (s.length() == 1) {
            char x = s.charAt(0);
            return new CharacterLiteral(pos(), x, sym.CHARACTER_LITERAL);
        }
        else {
            eq.enqueue(ErrorInfo.LEXICAL_ERROR,
                       "Illegal character literal \'" + s + "\'", pos(s.length()));
            return new CharacterLiteral(pos(), '\0', sym.CHARACTER_LITERAL);
        }
    }

    private Token boolean_lit(boolean x) {
        return new BooleanLiteral(pos(), x, sym.BOOLEAN_LITERAL);
    }

    private Token null_lit() {
        return new NullLiteral(pos(), sym.NULL_LITERAL);
    }

    private Token string_lit(String text) { //TWEAK: string arg
        return new StringLiteral(pos(text.length()), text,
                                 sym.STRING_LITERAL);
    }

	private int comment_count = 0;
	
    private String chop(int i, int j) {
        return yytext().substring(i,yylength()-j);
    }

    private String chop(int j) {
        return chop(0, j);
    }

    private String chop() {
        return chop(0, 1);
    }
%}

%component java, java_decl //actually identical
%component aspect, aspect_decl //actually identical
%component pointcut, pointcut2 //actually identical

//%component dot_class

%component character
%component string

%component nestablecomment
%component nonnestablecomment

%start java

%%

%%embed
%name nestable_comment
%host pointcut, pointcut2, aspect, aspect_decl, java, java_decl
%guest nestablecomment
%start START_NESTABLE_COMMENT
%end END_COMMENT
%pair START_NESTABLE_COMMENT, END_COMMENT

%%embed
%name nonnestable_comment
%host pointcut, pointcut2, aspect, aspect_decl, java, java_decl
%guest nonnestablecomment
%start START_NONNESTABLE_COMMENT
%end END_COMMENT

%%embed
%name string
%host pointcut, pointcut2, aspect, aspect_decl, java, java_decl
%guest string
%start START_STRING
%end END_STRING

%%embed
%name char
%host aspect, aspect_decl, java, java_decl
%guest character
%start START_CHARACTER
%end END_CHARACTER

//NB: currently based on paper rather than existing parser

%%embed
%name aspect_decl
%host java, aspect
%guest aspect_decl
%start ASPECT
%end LBRACE

%%embed
%name aspect
%host java, aspect
%guest aspect
%start %aspect_decl%
%end RBRACE
%pair LBRACE, RBRACE
%pair %pointcut2%, RBRACE

//NB: don't have to worry about ".class" because it will appear as CLASS_FIELD
//TODO-AC: paper says it does apply in aspect, implementation says it doesn't
//NB: going with the implementation because Eric says it's better in practice
//NB: ajc has a weird mix where proceed() is a keyword (if there is no local def) but proceed is not.
/*
//NB: this is only useful if the class meta-token is actually used
%%embed
%name dot_class
%host java, java_decl, aspect, aspect_decl
%guest dot_class
%start DOT
%end [END_DOT_CLASS CLASS_FIELD]

%%embed
%name java_decl
%host java, aspect
%guest java_decl
%start [CLASS INTERFACE]
%end LBRACE

%%embed
%name java
%host java, aspect
%guest java
%start %java_decl%
%end RBRACE
%pair LBRACE, RBRACE
*/

%%embed
%name perclause
%host aspect_decl
%guest pointcut
%start [PERCFLOW PERCFLOWBELOW PERTARGET PERTHIS] LPAREN
%end RPAREN
%pair LPAREN, RPAREN

%%embed
%name declare
%host aspect
%guest pointcut
%start DECLARE
%end SEMICOLON

%%embed
%name pointcut
%host java, aspect
%guest pointcut
%start POINTCUT
%end SEMICOLON

%%embed
%name advice
%host aspect
%guest pointcut2
%start [BEFORE AFTER AROUND]
%end LBRACE

%%embed
%name ifexpr
%host pointcut, pointcut2
%guest aspect //TWEAK: was pointcutifexpr
%start PC_IF LPAREN
%end RPAREN
%pair LPAREN, RPAREN

//// ADDED TO MATCH ORIGINAL BEHAVIOUR /////////////////////////////////////////

/*

//These embeddings don't really make sense, but otherwise we behave differently
//on UnterminatedDeclareErrorCE.java

%%embed
%name pointcut_java_decl
%host pointcut
%guest java_decl
%start [CLASS INTERFACE]
%end LBRACE

%%embed
%name pointcut_java
%host pointcut
%guest java
%start %java_decl%
%end RBRACE
%pair LBRACE, RBRACE

*/