//TODO-AC: many of the catchall errors are recoverable - either return a bad token which will be discarded or just discard and make a note.
//based on http://jflex.de/manual.html

package metalexer;

import static metalexer.ComponentParser.Terminals.*;

import beaver.Symbol;
import beaver.Scanner;

%%

//general header info
%public
%final
%class ComponentScanner

//required for beaver compatibility
%extends Scanner
%unicode
%function nextToken
%type Symbol
%yylexthrow Scanner.Exception

//for debugging - track line and column
%line
%column

%init{
  saveStateAndTransition(INSIDE_HEADER_SECTION);
  blobBuf = new StringBuffer();
  markStartPosition();
%init}

%{
  //// Returning symbols ///////////////////////////////////////////////////////

  //Create a symbol using the current line and column number, as computed by JFlex
  //No attached value
  //Symbol is assumed to start and end on the same line
  //e.g. symbol(SEMICOLON)
  private Symbol symbol(short type) {
    return symbol(type, null);
  }
  
  //Create a symbol using the current line and column number, as computed by JFlex
  //Attached value gives content information
  //Symbol is assumed to start and end on the same line
  //e.g. symbol(IDENTIFIER, "x")
  private Symbol symbol(short type, Object value) {
    //NB: JFlex is zero-indexed, but we want one-indexed
    int startLine = yyline + 1;
    int startCol = yycolumn + 1;
    int endLine = startLine;
    int endCol = startCol + yylength() - 1;
    return symbol(type, value, startLine, startCol, endLine, endCol);
  }
  
  //Create a symbol using explicit position information (one-indexed)
  private Symbol symbol(short type, Object value, int startLine, int startCol, int endLine, int endCol) {
    int startPos = Symbol.makePosition(startLine, startCol);
    int endPos = Symbol.makePosition(endLine, endCol);
    return new Symbol(type, startPos, endPos, value);
  }
  
  //// Position ////////////////////////////////////////////////////////////////
  
  //records the position of a symbol
  private static class PositionRecord {
      int startLine = -1;
      int startCol = -1;
      int endLine = -1;
      int endCol = -1;
  }
  
  //the position of the current symbol
  private PositionRecord pos = new PositionRecord();
  
  //populate the start line and column fields of the Position record with
  //values from JFlex
  private void markStartPosition() {
    //correct to one-indexed
    pos.startLine = yyline + 1;
    pos.startCol = yycolumn + 1;
  }
  
  //populate the start line and column fields of the Position record with
  //values from JFlex
  private void markEndPosition() {
    //correct to one-indexed
    pos.endLine = yyline + 1;
    pos.endCol = (yycolumn + 1) + yylength() - 1;
  }
  
  //like symbol(type), but uses the position stored in pos rather than
  //the position computed by JFlex
  private Symbol symbolFromMarkedPositions(short type) {
    return symbolFromMarkedPositions(type, null);
  }
  
  //like symbol(type, value), but uses the position stored in pos rather than
  //the position computed by JFlex
  private Symbol symbolFromMarkedPositions(short type, Object value) {
    return symbol(type, value, pos.startLine, pos.startCol, pos.endLine, pos.endCol);
  }
  
  //like symbol(type), but uses the start position stored in pos rather than
  //the start position computed by JFlex and an explicit length param rather
  //than yylength
  private Symbol symbolFromMarkedStart(short type, int length) {
    return symbolFromMarkedStart(type, null, length);
  }
  
  //like symbol(type, value), but uses the start position stored in pos rather than
  //the start position computed by JFlex and an explicit length param rather
  //than yylength
  private Symbol symbolFromMarkedStart(short type, Object value, int length) {
    return symbol(type, value, pos.startLine, pos.startCol, pos.startLine, pos.startCol + length - 1);
  }
  
  //// Errors //////////////////////////////////////////////////////////////////
  
  //throw an exceptions with position information from JFlex
  private void error(String msg) throws Scanner.Exception {
    //correct to one-indexed
    throw new Scanner.Exception(yyline + 1, yycolumn + 1, msg);
  }
  
  //throw an exceptions with position information from JFlex
  //columnOffset is added to the column
  private void error(String msg, int columnOffset) throws Scanner.Exception {
  //correct to one-indexed
    throw new Scanner.Exception(yyline + 1, yycolumn + 1 + columnOffset, msg);
  }
  
  //// State transitions ///////////////////////////////////////////////////////
  
  //stack entry: stack identifier + symbol position
  private static class StateRecord {
    int stateNum;
    PositionRecord pos;
    
    StateRecord(int stateNum, PositionRecord pos) {
        this.stateNum = stateNum;
        this.pos = pos;
    }
  }
  
  //most of our states are used for bracketing
  //this gives us a way to nest bracketing states
  private java.util.Stack<StateRecord> stateStack = new java.util.Stack<StateRecord>();
  
  void saveStateAndTransition(int newState) {
    stateStack.push(new StateRecord(yystate(), pos));
    pos = new PositionRecord();
    yybegin(newState);
  }
  
  void restoreState() {
    StateRecord rec = stateStack.pop();
    yybegin(rec.stateNum);
    pos = rec.pos;
  }
  
  //// Comment nesting /////////////////////////////////////////////////////////
  
  //number of '*/'s
  private int nestingDepth = 0;
  
  //// Blob accumulation ///////////////////////////////////////////////////////
  
  //for accumulating the contents of a string literal, comment, action, etc
  private StringBuffer blobBuf = new StringBuffer();
%}

LineTerminator = \r|\n|\r\n
OtherWhiteSpace = [ \t\f]

Letter = [a-zA-Z]
Digit = [0-9]
Identifier = {Letter} (_ | {Letter} | {Digit})*
QualifiedIdentifier = {Identifier} (\. {Identifier})+
Number = 0 | [1-9] {Digit}*

SectionSeparator = "%%"
GroupSeparator = "%:"

Quote = \"

//TODO-AC: "a backslash followed by any other unicode character that stands for this character."
HexDigit = {Digit} | [a-fA-F]
EscapeSequence = \\ ( [nrtfb] | x {HexDigit}{2} | u {HexDigit}{1,4} | [0-3]? [0-7]? [0-7] | [\|\(\)\{\}\[\]\<\>\\\.\*\+\?\^\$\/\.\"\~\!\-])

Comment = "//"[^\r\n]*
OpenBracketComment = "/*"
CloseBracketComment = "*/"

OpenDeclRegion = "%{"
CloseDeclRegion = "%}"

OpenInitRegion = "%init{"
CloseInitRegion = "%init}"

OpenAppendRegion = "%append{"
CloseAppendRegion = "%append}"

OpenAppendWithStartDelimRegion = "%appendWithStartDelim{"
CloseAppendWithStartDelimRegion = "%appendWithStartDelim}"

OpenAction = "{:"
CloseAction = ":}"

OpenCurlyBracket = \{
CloseCurlyBracket = \}

OpenAngleBracket = \<
CloseAngleBracket = \>

NonMeta = [^\|\(\)\{\}\[\]\<\>\\\.\*\+\?\^\$\/\.\"\~\!\-]

Any = . | \n

//handles comments
DirectiveLookahead = !({Letter} | {Digit} | _) //NB: use not (instead of a class) because it also catches EOF

//ok to not handle comments - JFlex doesn't either
MacroLookahead = {OtherWhiteSpace}* {Identifier} {OtherWhiteSpace}* {CloseCurlyBracket}
RepetitionLookahead = {OtherWhiteSpace}* {Number} ({OtherWhiteSpace}* "," {OtherWhiteSpace}* {Number})? {OtherWhiteSpace}* {CloseCurlyBracket}

//TODO-AC: doesn't handle comments
DeleteLookahead = ({LineTerminator} | {OtherWhiteSpace})+ \<

//TODO-AC: places that still can't have comments
//  Between BOL and directive / macro name
//  Between "%delete" and state list

//within the header section (serves as start state)
%xstate INSIDE_HEADER_SECTION
//within the lexical rule section
%xstate INSIDE_RULE_SECTION
//within a bracket comment (i.e. /*) - nestable
%xstate INSIDE_BRACKET_COMMENT
//within a string literal
%xstate INSIDE_STRING
//within a declaration region (i.e. %{)
%xstate INSIDE_DECL_REGION
//within an init region (i.e. %init{)
%xstate INSIDE_INIT_REGION
//within an append region (i.e. %append{)
%xstate INSIDE_APPEND_REGION
//within an append region (i.e. %appendWithStartDelim{)
%xstate INSIDE_APPEND_WITH_START_DELIM_REGION
//within a pair of curly brackets
%xstate INSIDE_MACRO
%xstate INSIDE_REPETITION_SPEC
%xstate INSIDE_ACTION
//within a pair of angle brackets
%xstate INSIDE_ANGLE_BRACKETS
//within a directive with identifier arguments (i.e. %component, %state, %xstate)
%xstate INSIDE_IDENTIFIER_DIRECTIVE
//within a directive with string arguments (i.e. the others)
%xstate INSIDE_STRING_DIRECTIVE
//within a delete directive (i.e. %delete)
%xstate INSIDE_DELETE_DIRECTIVE
//within a pair of angle brackets following a delete directive
%xstate INSIDE_DELETE_ANGLE_BRACKETS
//looking for optional meta token name (i.e. identifier) after action
%xstate INSIDE_META_TOKEN
//looking for an LCURLY to open a rule group
%xstate INSIDE_OPEN_RULE_GROUP
//within a character class (i.e. square brackets)
%xstate INSIDE_CHARACTER_CLASS

%%
//TODO-AC: might want to allow comments inside code blobs
//definitiely not: INSIDE_STRING, INSIDE_MACRO, INSIDE_REPETITION_SPEC
//maybe: INSIDE_DECL_REGION, INSIDE_INIT_REGION, INSIDE_APPEND_REGION, INSIDE_APPEND_WITH_START_DELIM_REGION, INSIDE_ACTION
<INSIDE_HEADER_SECTION, INSIDE_RULE_SECTION, INSIDE_DELETE_DIRECTIVE,
 INSIDE_META_TOKEN, INSIDE_ANGLE_BRACKETS, INSIDE_DELETE_ANGLE_BRACKETS,
 INSIDE_IDENTIFIER_DIRECTIVE, INSIDE_STRING_DIRECTIVE, INSIDE_OPEN_RULE_GROUP> {
    //single-line comments
    {Comment} { 
        /* ignore */
        //return symbol(COMMENT, yytext());
    }
    
    //start multiline comment
    {OpenBracketComment} {
        saveStateAndTransition(INSIDE_BRACKET_COMMENT);
        //blobBuf = new StringBuffer(yytext());
        //markStartPosition();
        nestingDepth++;
    }
}

<INSIDE_HEADER_SECTION> {
    //whitespace
    {LineTerminator} { /* ignore */ }
    {OtherWhiteSpace} { /* ignore */ }
    
    //start declaration region
    {OpenDeclRegion} {
        saveStateAndTransition(INSIDE_DECL_REGION);
        blobBuf = new StringBuffer(yytext());
        markStartPosition();
    }
    
    //start init region
    {OpenInitRegion} {
        saveStateAndTransition(INSIDE_INIT_REGION);
        blobBuf = new StringBuffer(yytext());
        markStartPosition();
    }
    
    //start append region
    {OpenAppendRegion} {
        saveStateAndTransition(INSIDE_APPEND_REGION);
        blobBuf = new StringBuffer(yytext());
        markStartPosition();
    }
    {OpenAppendWithStartDelimRegion} {
        saveStateAndTransition(INSIDE_APPEND_WITH_START_DELIM_REGION);
        blobBuf = new StringBuffer(yytext());
        markStartPosition();
    }
    
    {OpenCurlyBracket} / {MacroLookahead} {
        saveStateAndTransition(INSIDE_MACRO);
        return symbol(OPEN_MACRO);
    }
    
    {OpenCurlyBracket} / {RepetitionLookahead} {
        saveStateAndTransition(INSIDE_REPETITION_SPEC);
        return symbol(OPEN_REPETITION_SPEC);
    }
    
    //no-arg directives
    ^ {OtherWhiteSpace}* "%helper" / {DirectiveLookahead} {
        return symbol(HELPER_DIRECTIVE);
    }
    
    //identifier directives
    ^ {OtherWhiteSpace}* "%component" / {DirectiveLookahead} {
        saveStateAndTransition(INSIDE_IDENTIFIER_DIRECTIVE);
        return symbol(COMPONENT_DIRECTIVE);
    }
    ^ {OtherWhiteSpace}* "%state" / {DirectiveLookahead} {
        saveStateAndTransition(INSIDE_IDENTIFIER_DIRECTIVE);
        return symbol(STATE_DIRECTIVE);
    }
    ^ {OtherWhiteSpace}* "%xstate" / {DirectiveLookahead} {
        saveStateAndTransition(INSIDE_IDENTIFIER_DIRECTIVE);
        return symbol(XSTATE_DIRECTIVE);
    }
    ^ {OtherWhiteSpace}* "%start" / {DirectiveLookahead} {
        saveStateAndTransition(INSIDE_IDENTIFIER_DIRECTIVE);
        return symbol(START_DIRECTIVE);
    }
    
    //string directives
    ^ {OtherWhiteSpace}* "%extern" / {DirectiveLookahead} {
        saveStateAndTransition(INSIDE_STRING_DIRECTIVE);
        return symbol(EXTERN_DIRECTIVE);
    }
    ^ {OtherWhiteSpace}* "%import" / {DirectiveLookahead} {
        saveStateAndTransition(INSIDE_STRING_DIRECTIVE);
        return symbol(IMPORT_DIRECTIVE);
    }
    ^ {OtherWhiteSpace}* "%initthrow" / {DirectiveLookahead} {
        saveStateAndTransition(INSIDE_STRING_DIRECTIVE);
        return symbol(INITTHROW_DIRECTIVE);
    }
    ^ {OtherWhiteSpace}* "%lexthrow" / {DirectiveLookahead} {
        saveStateAndTransition(INSIDE_STRING_DIRECTIVE);
        return symbol(LEXTHROW_DIRECTIVE);
    }
    
    //invalid directives
    ^ {OtherWhiteSpace}* "%" { error("Invalid directive"); }
    
    //for macro declarations
    //NB: beginning of line so that it doesn't interfere with patterns
    ^ {OtherWhiteSpace}* {Identifier} { return symbol(IDENTIFIER, yytext()); }
    
    //end of section
    ^ {OtherWhiteSpace}* {SectionSeparator} { 
        restoreState();
        saveStateAndTransition(INSIDE_RULE_SECTION);
        return symbol(SECTION_SEPARATOR);
    }
}

<YYINITIAL> {
    {Any} {
        error("Unexpected character: " + yytext());
    }
    <<EOF>> {
        return symbol(EOF);
    }
}

<INSIDE_RULE_SECTION> {
    //whitespace
    {LineTerminator} { /* ignore */ }
    {OtherWhiteSpace} { /* ignore */ }
    
    {OpenAngleBracket} {
        saveStateAndTransition(INSIDE_ANGLE_BRACKETS);
        return symbol(LANGLE);
    }
    
    //lookahead ensures that "%delete" isn't a prefix (e.g. "%deleted") and that we're not in a regex (since no <)
    ^ {OtherWhiteSpace}* "%delete" / {DeleteLookahead} {
        saveStateAndTransition(INSIDE_DELETE_DIRECTIVE);
        return symbol(DELETE_DIRECTIVE);
    }
    
    {GroupSeparator} { return symbol(GROUP_SEPARATOR); }
    {SectionSeparator}inherit {
        saveStateAndTransition(INSIDE_IDENTIFIER_DIRECTIVE);
        return symbol(INHERIT_SECTION_SEPARATOR);
    }
}

<INSIDE_DELETE_DIRECTIVE> {
    //whitespace
    {LineTerminator} { restoreState(); return symbol(DELETE_TERMINATOR); }
    {OtherWhiteSpace} { /* ignore */ }
    
    {OpenAngleBracket} {
        saveStateAndTransition(INSIDE_DELETE_ANGLE_BRACKETS);
        return symbol(LANGLE);
    }
}

<INSIDE_RULE_SECTION, INSIDE_DELETE_DIRECTIVE> {
    {OpenCurlyBracket} / {MacroLookahead} {
        saveStateAndTransition(INSIDE_MACRO);
        return symbol(OPEN_MACRO);
    }
    
    {OpenCurlyBracket} / {RepetitionLookahead} {
        saveStateAndTransition(INSIDE_REPETITION_SPEC);
        return symbol(OPEN_REPETITION_SPEC);
    }
}

<INSIDE_RULE_SECTION> {
    {OpenAction} {
        saveStateAndTransition(INSIDE_ACTION);
        blobBuf = new StringBuffer(yytext());
        markStartPosition();
    }
    
    //for ending state rule-groups
    {CloseCurlyBracket} { return symbol(CLOSE_RULE_GROUP); }
}

<INSIDE_HEADER_SECTION, INSIDE_RULE_SECTION, INSIDE_DELETE_DIRECTIVE> {
    \( { return symbol(LPAREN); }
    \) { return symbol(RPAREN); }
    \[ { saveStateAndTransition(INSIDE_CHARACTER_CLASS); return symbol(LSQUARE); }
    \] { return symbol(RSQUARE); }
    
    \^ { return symbol(BEGINNING_OF_LINE); }
    \$ { return symbol(END_OF_LINE); }
    
    \! { return symbol(NOT); }
    \~ { return symbol(UPTO); }
    
    \* { return symbol(STAR); }
    \+ { return symbol(PLUS); }
    \? { return symbol(OPT); }
    
    \- { return symbol(DASH); }
    
    \/ { return symbol(SLASH); }
    \| { return symbol(ALT); }
    
    \. { return symbol(DOT); }
    
    "=" { return symbol(ASSIGN); }
    
    "<<ANY>>" { return symbol(ANY_PATTERN); }
    "<<EOF>>" { return symbol(EOF_PATTERN); }
    
    {EscapeSequence} { return symbol(ESCAPE_SEQUENCE, yytext()); }
    \\{Any} { return symbol(NON_META, yytext().substring(1)); }
    \\ { error("Incomplete escape sequence"); }
    
    {Quote} {
        saveStateAndTransition(INSIDE_STRING);
        blobBuf = new StringBuffer(yytext());
        markStartPosition();
    }
    
    //safe fallback for patterns
    {NonMeta} { return symbol(NON_META, yytext()); }
    
    //catchall - error
    //safe because INSIDE_CURLY_BRACKETS handles actions
    {Any} {
        error("Unexpected character: " + yytext());
        yybegin(YYINITIAL); //central handling of EOF
    }
}

<INSIDE_CHARACTER_CLASS> {
    \^ { return symbol(CHAR_CLASS_NEGATE); }
    \- { return symbol(DASH); }
    \] { restoreState(); return symbol(RSQUARE); }
    
    {EscapeSequence} { return symbol(ESCAPE_SEQUENCE, yytext()); }
    \\{Any} { return symbol(CHAR_CLASS_CHAR, yytext().substring(1)); }
    \\ { error("Incomplete escape sequence"); }
    
    {Any} { return symbol(CHAR_CLASS_CHAR, yytext()); }
}

<INSIDE_HEADER_SECTION, INSIDE_RULE_SECTION> {
    <<EOF>> {
        yybegin(YYINITIAL); //central handling of EOF
    }
}

<INSIDE_DELETE_DIRECTIVE> {
    <<EOF>> {
        yybegin(YYINITIAL); //central handling of EOF
        return symbol(DELETE_TERMINATOR);
    }
}

//remainder of declaration region
<INSIDE_DECL_REGION> {
    %{CloseDeclRegion} { blobBuf.append(yytext().substring(1)); }
    {CloseDeclRegion} {
        blobBuf.append(yytext());
        markEndPosition();
        Symbol sym = symbolFromMarkedPositions(DECL_REGION, blobBuf.toString());
        restoreState();
        return sym;
    }
    {Any} { blobBuf.append(yytext()); }
    <<EOF>> {
        yybegin(YYINITIAL); 
        error("Unterminated declaration region: '" + blobBuf + "'");
    }
}

//remainder of init region
<INSIDE_INIT_REGION> {
    %{CloseInitRegion} { blobBuf.append(yytext().substring(1)); }
    {CloseInitRegion} {
        blobBuf.append(yytext());
        markEndPosition();
        Symbol sym = symbolFromMarkedPositions(INIT_REGION, blobBuf.toString());
        restoreState();
        return sym;
    }
    {Any} { blobBuf.append(yytext()); }
    <<EOF>> {
        yybegin(YYINITIAL); 
        error("Unterminated init region: '" + blobBuf + "'");
    }
}

//remainder of append region
<INSIDE_APPEND_REGION> {
    %{CloseAppendRegion} { blobBuf.append(yytext().substring(1)); }
    {CloseAppendRegion} {
        blobBuf.append(yytext());
        markEndPosition();
        Symbol sym = symbolFromMarkedPositions(APPEND_REGION, blobBuf.toString());
        restoreState();
        return sym;
    }
    {Any} { blobBuf.append(yytext()); }
    <<EOF>> {
        yybegin(YYINITIAL); 
        error("Unterminated append region: '" + blobBuf + "'");
    }
}

//remainder of appendWithStartDelim region
<INSIDE_APPEND_WITH_START_DELIM_REGION> {
    %{CloseAppendWithStartDelimRegion} { blobBuf.append(yytext().substring(1)); }
    {CloseAppendWithStartDelimRegion} {
        blobBuf.append(yytext());
        markEndPosition();
        Symbol sym = symbolFromMarkedPositions(APPEND_WITH_START_DELIM_REGION, blobBuf.toString());
        restoreState();
        return sym;
    }
    {Any} { blobBuf.append(yytext()); }
    <<EOF>> {
        yybegin(YYINITIAL); 
        error("Unterminated append region: '" + blobBuf + "'");
    }
}

//remainder of string literal (i.e. after initial single quote)
<INSIDE_STRING> {
    {Quote} {
        blobBuf.append(yytext());
        markEndPosition();
        Symbol sym = symbolFromMarkedPositions(STRING, blobBuf.toString());
        restoreState();
        return sym;
    }
    {EscapeSequence} { blobBuf.append(yytext()); }
    \\{Any} { blobBuf.append(yytext().substring(1)); }
    \\ {
        yybegin(YYINITIAL);
        error("Incomplete escape sequence");
    }
    {LineTerminator} {
        yybegin(YYINITIAL); 
        error("Unterminated string literal: '" + blobBuf + "'");
    }
    . { blobBuf.append(yytext()); }
    <<EOF>> {
        yybegin(YYINITIAL); 
        error("Unterminated string literal: '" + blobBuf + "'");
    }
}

//continue multiline comment
<INSIDE_BRACKET_COMMENT> {
    {OpenBracketComment} {
        //blobBuf.append(yytext());
        nestingDepth++;
    }
    {CloseBracketComment} { 
        //blobBuf.append(yytext());
        nestingDepth--;
        if(nestingDepth == 0) {
            //markEndPosition();
            //Symbol sym = symbolFromMarkedPositions(BRACKET_COMMENT, blobBuf.toString());
            restoreState();
            //return sym;
        }
    }
    {Any} {
        //blobBuf.append(yytext());
    }
    <<EOF>> {
        yybegin(YYINITIAL);
        //don't finish scanning if there's an unclosed comment
        if(nestingDepth != 0) {
            error(nestingDepth + " levels of comments not closed");
        }
    }
}

//NB: lookahead does not allow comments, so we don't either (as in JFlex)
<INSIDE_MACRO> {
    //whitespace
    {LineTerminator} { /*ignore*/ }
    {OtherWhiteSpace} { /*ignore*/ }
    
    //for macro names
    {Identifier} { return symbol(IDENTIFIER, yytext()); }
    
    {CloseCurlyBracket} {
        restoreState();
        return symbol(CLOSE_MACRO);
    }
    
    //catchall - error
    {Any} {
        error("Unexpected character in macro invocation: " + yytext());
        yybegin(YYINITIAL); //central handling of EOF
    }
}

//NB: lookahead does not allow comments, so we don't either (as in JFlex)
<INSIDE_REPETITION_SPEC> {
    //whitespace
    {LineTerminator} { /*ignore*/ }
    {OtherWhiteSpace} { /*ignore*/ }
    
    //for repetition quantities
    {Number} { return symbol(NUMBER, yytext()); }
    
    //for separating repetition quantities
    , { return symbol(COMMA); }
    
    {CloseCurlyBracket} {
        restoreState();
        return symbol(CLOSE_REPETITION_SPEC);
    }
    
    //catchall - error
    {Any} {
        error("Unexpected character in repetition specification: " + yytext());
        yybegin(YYINITIAL); //central handling of EOF
    }
}

<INSIDE_ACTION> {
    %{CloseAction} { blobBuf.append(yytext().substring(1)); }
    {CloseAction} { 
        blobBuf.append(yytext());
        markEndPosition();
        Symbol sym = symbolFromMarkedPositions(ACTION, blobBuf.toString());
        restoreState(); //NB: restore before transitioning
        saveStateAndTransition(INSIDE_META_TOKEN);
        return sym;
    }
    {Any} {
        blobBuf.append(yytext());
    }
    <<EOF>> {
        yybegin(YYINITIAL);
    }
}

<INSIDE_META_TOKEN> {
    //whitespace
    {LineTerminator} { restoreState(); /* ignore */ }
    {OtherWhiteSpace} { /* ignore */ }
    
    //for meta-token types
    {Identifier} { return symbol(IDENTIFIER, yytext()); }
    
    //catchall - error
    {Any} {
        error("Unexpected character in meta token specification: " + yytext());
        yybegin(YYINITIAL); //central handling of EOF
    }
    <<EOF>> {
        yybegin(YYINITIAL);
    }
}

//i.e. don't do this if INSIDE_DELETE_ANGLE_BRACKETS
<INSIDE_ANGLE_BRACKETS> {
    {CloseAngleBracket} {
        restoreState();
        saveStateAndTransition(INSIDE_OPEN_RULE_GROUP);
        return symbol(RANGLE);
    }
}

<INSIDE_DELETE_ANGLE_BRACKETS> {
    {CloseAngleBracket} {
        restoreState();
        return symbol(RANGLE);
    }
}

<INSIDE_DELETE_ANGLE_BRACKETS, INSIDE_ANGLE_BRACKETS> {
    //whitespace
    {LineTerminator} { /* ignore */ }
    {OtherWhiteSpace} { /* ignore */ }
    
    //for state names
    {Identifier} { return symbol(IDENTIFIER, yytext()); }
    
    //for separating state names
    , { return symbol(COMMA); }
    
    //catchall - error
    {Any} {
        error("Unexpected character between angle brackets: " + yytext());
        yybegin(YYINITIAL); //central handling of EOF
    }
    <<EOF>> {
        error("Unterminated angle brackets"); //TODO-AC: could attach position
        yybegin(YYINITIAL); //central handling of EOF
    }
}

<INSIDE_OPEN_RULE_GROUP> {
    //whitespace
    {LineTerminator} { /* ignore */ }
    {OtherWhiteSpace} { /* ignore */ }
    
    {OpenCurlyBracket} {
        restoreState();
        return symbol(OPEN_RULE_GROUP);
    }
    
    //catchall - error
    {Any} {
        error("Expecting '{', found: " + yytext());
        yybegin(YYINITIAL); //central handling of EOF
    }
    <<EOF>> {
        error("No group associated with state list");
        yybegin(YYINITIAL); //central handling of EOF
    }
}

<INSIDE_IDENTIFIER_DIRECTIVE> {
    //whitespace
    {LineTerminator} { restoreState(); /* ignore */ }
    {OtherWhiteSpace} { /* ignore */ }
    
    //for state names
    {Identifier} { return symbol(IDENTIFIER, yytext()); }
    {QualifiedIdentifier} { return symbol(QUALIFIED_IDENTIFIER, yytext()); }
    
    , { return symbol(COMMA); }
    
    //catchall - error
    {Any} {
        error("Unexpected character in identifier list: " + yytext());
        yybegin(YYINITIAL); //central handling of EOF
    }
    <<EOF>> {
        yybegin(YYINITIAL); //central handling of EOF
    }
}

<INSIDE_STRING_DIRECTIVE> {
    //whitespace
    {LineTerminator} { restoreState(); /* ignore */ }
    {OtherWhiteSpace} { /* ignore */ }
    
    //for strings
    {Quote} {
        saveStateAndTransition(INSIDE_STRING);
        blobBuf = new StringBuffer(yytext());
        markStartPosition();
    }
    
    , { return symbol(COMMA); }
    
    //catchall - error
    {Any} {
        error("Unexpected character in string list: " + yytext());
        yybegin(YYINITIAL); //central handling of EOF
    }
    <<EOF>> {
        yybegin(YYINITIAL); //central handling of EOF
    }
}
