/* bintree.c - Binary tree structure */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include "bintree.h"

typedef struct BinaryTreeCell {
    void *info;
    struct BinaryTreeCell *left, *right;
} BinaryTreeCell;

typedef BinaryTreeCell BTC;

BTC *new_BTC (void *info, BTC *left, BTC *right);
static void BTC_delete (BTC *cell);
static void BTC_lookup (BTC *root, void *elem, comparison_function_t cmpfct,
		    BTC **found, int *code);
static int BTC_insert (BTC *cell, comparison_function_t cmpfct, void *elem);
static void BTC_iterate (BTC *cell, void (*f) (void *));


/* static int valid_tree (const BinaryTree *tree) */
static int valid_tree ( BinaryTree *tree) 
{
    if (!tree || tree->nbelem < 0 || tree->cmpfct == NULL)
	return 0;
    if ((tree->nbelem == 0) != (tree->root == NULL))
	return 0;
    return 1;
}


void BinaryTree_initialize (BinaryTree *tree, comparison_function_t f)
{
    assert (tree);
    assert (f);
    tree->nbelem = 0;
    tree->cmpfct = f;
    tree->root = NULL;
    assert (valid_tree(tree));
}


void BinaryTree_empty (BinaryTree *tree)
{
    assert (valid_tree(tree));
    BTC_delete (tree->root);
    tree->root = NULL;
    tree->nbelem = 0;
    assert (valid_tree(tree));
}


void BTC_delete (BTC *cell)
{
    if (cell) {
	BTC_delete (cell->left);
	BTC_delete (cell->right);
	free (cell->info);
	free (cell);
    }
}


int BinaryTree_insert (BinaryTree *tree, void *elem)
{
    BTC *cell;
    int new_element;

    assert (valid_tree(tree));
    assert (elem);
    if (tree->root == NULL) {  /* if tree is empty */
	tree->root = new_BTC(elem, NULL, NULL);
	tree->nbelem = 1;
	new_element = 1;
    }
    else  /* tree is not empty */
	new_element = BTC_insert(tree->root, tree->cmpfct, elem);
    assert (valid_tree(tree));
    return new_element;
}


void BTC_lookup (BTC *root, void *elem, comparison_function_t cmpfct,
		    BTC **found, int *code)
/* Codes returned in *code:
   0 = '*found' points to cell that contains 'elem';
   1 = '*found' points to cell to which 'elem' would be the left child;
   2 = '*found' points to cell to which 'elem' would be the right child;
   3 = root is NULL (i.e. tree is empty) and '*found' is unchanged.
*/
{
    BTC *parent = NULL, *current;
    int n;

    current = root;
    while (current != NULL) {
	n = cmpfct(elem, current->info);
	switch (n) {
	    case 0:  /* equality */
	        *found = current;
	        *code = 0;
	        return;
	    case -1:  /* elem is smaller than current */
		parent = current;
		current = current->left;
		*code = 1;
		break;
	    case +1:  /* elem is greater than current */
		parent = current;
		current = current->right;
		*code = 2;
		break;
	    default:
		assert (0);
	}
    }
    /* elem not found in tree */
    if (parent) {
	*found = parent;
	/* *code already contains the appropriate value */
	return;
    }
    /* parent is NULL, hence root is NULL */
    *found = NULL;
    *code = 3;
}


int BTC_insert (BTC *cell, comparison_function_t cmpfct, void *elem)
{
    BTC *found;
    int code;

    assert (cell);  /* we expect a non-empty tree */
    BTC_lookup (cell, elem, cmpfct, &found, &code);
    switch (code) {
	case 0:  /* elem is in the tree */
	    return 0;
	case 1:  /* found points to parent; give it new left child */
	    assert (found->left == NULL);
	    found->left = new_BTC(elem, NULL, NULL);
	    return 1;
	case 2:  /* found points to parent; give it new right child */
	    assert (found->right == NULL);
	    found->right = new_BTC(elem, NULL, NULL);
	    return 1;
    }
    assert (0);
    return 0;
}


int BinaryTree_cardinal (BinaryTree *tree)
{
    assert (valid_tree(tree));
    return tree->nbelem;
}


void BinaryTree_iterate (BinaryTree *tree, void (*f) (void *))
{
    assert (valid_tree(tree));
    BTC_iterate (tree->root, f);
    assert (valid_tree(tree));
}


void BTC_iterate (BTC *cell, void (*f) (void *))
{
    if (cell) {
        BTC_iterate (cell->left, f);
	f (cell->info);
	BTC_iterate (cell->right, f);
    }
}


BTC *new_BTC (void *info, BTC *left, BTC *right)
{
    BTC *cell;

    cell = (BTC *) malloc(sizeof(BTC));
    assert (cell);
    cell->info = info;
    cell->left = left;
    cell->right = right;
    return cell;
}

