#include <malloc.h>
#include <assert.h>
#include <errno.h>
#include <stdio.h>
#ifndef SEQUENT
#include <stdlib.h>
#endif

#include "main.h"
#include "sparse.h"

#ifdef UNIX 
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#endif

/* prototypes we can't put in main.h */
#ifdef OLDC
SparseMatrix_t *buildMatrix();
#else
SparseMatrix_t *buildMatrix(void);
#endif

/* globals */
static long g_maxRow;
static long g_maxCol;
static long g_numElems;
static long g_maxVal;

#ifdef OLDC
int main(argc, argv)
int argc;
char *argv[];
#else
int main(int argc, char *argv[])
#endif
{
  SparseMatrix_t  *sm;
  double           startTime, stopTime, *x; 
  long             i;

#ifndef RUNTIME
  printf("starting...\n\n");
#endif

#ifdef NEVER
  test1();
  test2();
#endif
  
  x = init(argc, argv);
  /* x = x = (double *) malloc( sizeof(double) * (g_maxCol + 1)); Rakesh*/
#ifndef RUNTIME
  printf("building matrix...\n");
#endif
  sm = buildMatrix();

#ifndef RUNTIME
  printf("Initial matrix has %ld elements (including RHS b).\n", 
         sparseNumElements(sm));
  printf("solving...\n");
#endif
  fflush(stdout);

  startTime = cputime(); 

#ifdef ScaleMatrix
  if ( sparseScale(sm) == False)
    error("scale failed in Main");
#endif

  if ( sparseFactor(sm) == False )
    error("factor failed in Main");

#ifdef SolveMatrix
  if ( sparseSolve(sm, x) == False )
    error("solve failed in Main");
#endif

  stopTime = cputime();

#ifndef RUNTIME
  printf("Solved matrix has %ld elements.\n", sparseNumElements(sm));
#endif
  /*printf("Total System and User Time: %f seconds.\n", stopTime - startTime);*/
#ifndef RUNTIME
#ifdef OutputSolution 
  printf("\nSolution vector x:\n");
  for (i = 0; i <= g_maxCol; i++)
    printf("  x[%d] = %f\n", i, x[i]);
#endif
#endif

  free(x);
  sparseFree(sm);

#ifndef RUNTIME
  printf("\ndone.\n");
#endif
  return 0;
}


/* ######################################################################## */
/*
 * init: 
 */

#ifdef OLDC
double *init(argc, argv)
int argc;
char *argv/* Rakesh[]: gives type-check probs*/;
#else
double *init(int argc, char **argv/* Rakesh gives type-check probs[]*/)
#endif
{
  double  *x;

  /* 
   * Initialize... 
   */
  
  g_maxRow   = MaxRow; 
  g_maxCol   = MaxCol;
  g_numElems = NumElements;
  g_maxVal   = MaxVal;

#ifdef SEED
  srandom(SEED);
#else

#ifdef UNIX
#ifdef TRUE_RANDOM
  { 
    time_t  t;

    t = time(NULL);
    srandom( (int) t ); 
  }
#else
  srandom(1);  /* same each time */ 
#endif
#else
  srandom(1);  /* if not on Unix, not sure how to get the time... */ 
#endif

#endif

  /*
   * Process any command-line args...
   */

  if (argc == 1) {
#ifndef RUNTIME
    printf("Assuming default matrix size and number of elements...\n");
    printf("( options are: -r maxrow -c maxcol -n numelems -v maxval )\n");
#endif
  }/*then*/
  else {  /* process args */

    argc--;
    while (argc > 0) {
 
      switch (argv[argc-1][1]) {
        case 'r': sscanf(argv[argc], "%ld", &g_maxRow); argc--; break;
        case 'c': sscanf(argv[argc], "%ld", &g_maxCol); argc--; break;
        case 'n': sscanf(argv[argc], "%ld", &g_numElems); argc--; break;
        case 'v': sscanf(argv[argc], "%ld", &g_maxVal); argc--; break;
#ifndef RUNTIME
        default: printf("unknown option.\n");
#endif
      }/*switch*/
      argc--;
    }/*while*/

  }/*else*/

#ifndef RUNTIME
  printf("params: %ld rows, %ld cols, %ld elements, range %ld..%ld.\n\n",
          g_maxRow+1, g_maxCol+1, g_numElems, -g_maxVal, g_maxVal);
#endif
  
  if (g_maxRow != g_maxCol)
    error("init: max row != max col (error, matrix must be square)");
  if (g_numElems < ((2*g_maxRow) + g_maxCol)) {
    char buf[128];
    sprintf(buf, "init: matrix must contain at least %ld elements", 
            (2*g_maxRow) + g_maxCol);
    error(buf);
  }/*then*/

  /* 
   * allocate an array big enough for solving matrix...
   */

  x = (double *) malloc( sizeof(double) * (g_maxCol + 1) );  /* zero-based */
  if (x == NULL)
    error("init: out of memory");

  return x;
}


/* ######################################################################## */
/*
 * buildMatrix 
 */

SparseMatrix_t *buildMatrix()
{
  SparseMatrix_t  *sm;
  long             i, row, col;
  double           val, junk;

  /* 
   * Let's build a sparse matrix whose size are controlled by the
   *   global variables... 
   */

  sm = sparseCreate();

  /* first, make sure every row has at least two elements... */
  for (i = 0; i <= g_maxRow; i++) {

    do {
      col = getRandomIntInRange(g_maxCol);
    } while (sparseGet(sm, i, col, &junk) == True);

    val = getRandomFloatInRange(-g_maxVal, g_maxVal);
    if (sparsePut(sm, i, col, val) == False)
      error("buildMatrix: sparsePut failed, out of memory");

    do {
      col = getRandomIntInRange(g_maxCol);
    } while (sparseGet(sm, i, col, &junk) == True);

    val = getRandomFloatInRange(-g_maxVal, g_maxVal);
    if (sparsePut(sm, i, col, val) == False)
      error("buildMatrix: sparsePut failed, out of memory");
  }/*for*/
      
  /* likewise, make sure every col has at least one element... */
  for (i = 0; i <= g_maxCol; i++) {

    do {
      row = getRandomIntInRange(g_maxCol);
    } while (sparseGet(sm, row, i, &junk) == True);

    val = getRandomFloatInRange(-g_maxVal, g_maxVal);
    if (sparsePut(sm, row, i, val) == False)
      error("buildMatrix: sparsePut failed, out of memory");
  }/*for*/
      
  /* now fill in the bulk of the matrix */
  for (i = 0; i < (g_numElems - ((2*g_maxRow) + g_maxCol + 3)); i++) {
    
    do {
      row = getRandomIntInRange(g_maxRow);
      col = getRandomIntInRange(g_maxCol);
    } while (sparseGet(sm, row, col, &junk) == True);

    val = getRandomFloatInRange(-g_maxVal, g_maxVal);
    if (sparsePut(sm, row, col, val) == False)
      error("buildMatrix: sparsePut failed, out of memory");
  }/*for*/

  /* create a RHS vector b... */
  for (i = 0; i <= g_maxRow; i++) {

    val = getRandomFloatInRange(-g_maxVal, g_maxVal);
    if (sparsePut(sm, i, g_maxCol+1, val) == False)
      error("buildMatrix: sparsePut failed, out of memory");
  }/*for*/

#ifdef NEVER 
  sparsePrint(sm);
#endif 

  return sm;
}


#ifdef OLDC
int  test2()
#else
void test2()
#endif
{
  SparseMatrix_t  *sm;
  long             i;
  double           x[1000];

  sm = sparseCreate();

  sparsePut(sm, 0, 0, 2000.0); 
  sparsePut(sm, 0, 1, 4.0); 
  sparsePut(sm, 0, 3, -3.0); 
  sparsePut(sm, 0, 4, 1.0); 
  sparsePut(sm, 0, 5, 48000.0); 
  sparsePut(sm, 1, 3, 2.0); 
  sparsePut(sm, 1, 4, 5.0); 
  sparsePut(sm, 1, 5, -20.0); 
  sparsePut(sm, 2, 2, 10.0); 
  sparsePut(sm, 2, 5, 100.0); 
  sparsePut(sm, 3, 1, 2.0); 
  sparsePut(sm, 3, 2, 1.0); 
  sparsePut(sm, 3, 4, -4.0); 
  sparsePut(sm, 3, 5, 24.0); 
  sparsePut(sm, 4, 0, 1.0); 
  sparsePut(sm, 4, 2, 3.0); 
  sparsePut(sm, 4, 5, 15.0); 

  sparsePut(sm, 0, 0, 2.0);    /* test reassignment... */
  sparsePut(sm, 0, 5, 48.0);

  sparsePrint(sm);
  printf("\n");

  if ( sparseScale(sm) == False)
    printf("*!*! sparse scale failed...\n");
  else {

    if ( sparseFactor(sm) == False )
      printf("*!*! sparse factor failed...\n");
    else {

      if ( sparseSolve(sm, x) == False )
        printf("*!*! attempt to solve matrix failed...\n");
      else {
        /* answer: -15, 9.424242, 10, -13.030303, 1.212121 */
        printf("solution vector x:\n");
        for (i = 0; i < sparseMaxCol(sm); i++)
          printf("  x[%ld] = %e.\n", i, x[i]);
      }/*else*/
    }/*else*/
  }/*else*/
  
  sparseFree(sm);
}


#ifdef OLDC
int  test1()
#else
void test1()
#endif
{
  SparseMatrix_t  *sm;
  long             i;
  double           x[1000];

  sm = sparseCreate();

  /* in this order, can be solved without factoring... */
  sparsePut(sm, 3, 3, 10.0);
  sparsePut(sm, 3, 4, 100.0); 
  sparsePut(sm, 2, 2, 9.0); 
  sparsePut(sm, 2, 3, 27.0);
  sparsePut(sm, 2, 4, 279.0);  
  sparsePut(sm, 1, 1, 2.0); 
  sparsePut(sm, 1, 4, 10.0);  
  sparsePut(sm, 0, 0, 3.0);
  sparsePut(sm, 0, 1, 9.0);  
  sparsePut(sm, 0, 2, -3.0); 
  sparsePut(sm, 0, 3, 9.0);  
  sparsePut(sm, 0, 4, 129.0);
  
#ifdef NEVER
  sparsePrint(sm);
  
  { SparseMatrix_t *dup;

    dup = sparseDup(sm);
    sparsePrint(dup);
    sparseFree(dup);
  }
#endif

  if ( sparseSolve(sm, x) == False )
    printf("*!*! attempt to solve matrix failed...\n");
  else {
    /* answer: -1, 5, 1, and 10. */
    printf("solution vector x:\n");
    for (i = 0; i < sparseMaxCol(sm); i++)
      printf("  x[%ld] = %e.\n", i, x[i]);
  }/*else*/
  
  sparseFree(sm);

  printf("\n");
  sm = sparseCreate();
  
  /* put in an order that requires factoring... */
  sparsePut(sm, 0, 3, 10.0); 
  sparsePut(sm, 0, 4, 100.0); 
  sparsePut(sm, 1, 2, 9.0); 
  sparsePut(sm, 1, 3, 27.0);
  sparsePut(sm, 1, 4, 279.0);  
  sparsePut(sm, 2, 1, 2.0); 
  sparsePut(sm, 2, 4, 10.0);  
  sparsePut(sm, 3, 0, 3.0);
  sparsePut(sm, 3, 1, 9.0);  
  sparsePut(sm, 3, 2, -3.0); 
  sparsePut(sm, 3, 3, 9.0);  
  sparsePut(sm, 3, 4, 129.0);

  sparseScale(sm);
  sparseFactor(sm);
  
  sparsePrint(sm);
  
  if ( sparseSolve(sm, x) == False )
    printf("*!*! attempt to solve matrix failed...\n");
  else {
    /* answer: -1, 5, 1, and 10. */
    printf("solution vector x:\n");
    for (i = 0; i < sparseMaxCol(sm); i++)
      printf("  x[%ld] = %e.\n", i, x[i]);
  }/*else*/

  printf("\n");
}


/* ######################################################################## */
/*
 * getRandomIntInRange: return an integer random number in the range 
 *   0 .. upper, inclusive.
 */

#ifdef OLDC
long getRandomIntInRange(upper)
long upper;
#else
long getRandomIntInRange(long upper)
#endif
{
  long  lower, num;

  lower = 0;
  num = random();

  /* get a float f such that 0 <= f < 1, expand to 0 <= f < top + 1, */
  /*   and then add in lower to get lower <= f < upper + 1, which is */
  /*   finally truncated and returned.                               */
  num = ((( ((double) num) / MaxRand ) * (upper - lower + 1) ) + lower );
  
  return num;
}


/* ######################################################################## */
/*
 * getRandomFloatInRange: return a floating-point random number in the 
 *   range lower .. upper, inclusive.
 * 
 * NOTE: if lower is negative, we assume you want an even mix of 
 *   negative and positive values.
 * NOTE: zero values are NEVER returned. 
 */

#ifdef OLDC
double getRandomFloatInRange(lower, upper)
long lower;
long upper;
#else
double getRandomFloatInRange(long lower, long upper)
#endif
{
  long    num, fraction;
  int     bit;
  double  d;

  /* first, get a non-zero random number... */
  do {
    num = getRandomIntInRange(upper);
  } while (num == 0);

  /* create a random floating point number... */
  fraction = getRandomIntInRange(999);
  d = ((double) num) + ( ((double) fraction) / 1000 );

  /* finally, if need negative values, flip a coin for sign bit... */ 
  if (lower < 0) {

    bit = random() & 01;  /* check lowest bit for sign... */
    if (bit == 1)
      d = -d;
  }/*if*/

#ifdef DEBUG
  if (d == 0.0) {
    fprintf(stderr, "float in range just got zero!\n");
    exit(0);
  }
#endif

  return d;
}


/* ######################################################################## */
/*
 * cputime: compute CPU time in seconds.
 */

#ifdef UNIX 

double cputime()
{
    struct rusage  usage;
    double         time;
    
    getrusage(RUSAGE_SELF, &usage);

    time = (double) usage.ru_utime.tv_sec
           + (usage.ru_utime.tv_usec / 1000000.0);
#if 0
           + (double) usage.ru_stime.tv_sec
           + (usage.ru_stime.tv_usec / 1000000.0);
#endif
           
    return time;
}

#else

double cputime()
{ 
    return -1.0;  /* otherwise don't know what to do... */
}

#endif


/* ######################################################################## */
/*
 * error: scream and die quickly, printing message first.
 */

#ifdef OLDC
int  error(msg)
char *msg;
#else
void error(char *msg)
#endif
{
  fprintf(stderr, "**ERROR: %s.\n", msg);  /* print error message      */ 
  if (errno != 0)                          /* system error msg as well */
    perror("Sys Error Msg:");

  exit(0);  /* die, but return with success to prevent core dump */
}

