/* sparse.c */

/* 
 * Assumptions:
 *   1. Row and cols are assumed to be in the range 0 .. MaxLong.
 *   2. Factor, solve, etc. assume that RHS b is stored in last column
 *      of matrix (thus, all the rows should contain an entry in the
 *      last column).
 *   3. matrix must be square to be factor or solved.
 *
 * Ref: "Sparse Matrix Techniques", K. Kundert, in "Circuit Analysis, 
 *   Simulation and Design", edited by A. Ruehli, North-Holland Pub,
 *   1986.
 *
 *
 * Todo: 
 *   1. row, col, etc. fields in SM should be shorts.
 *   2. better rescaling during sparseFactor; i.e., should look at matrix
 *      and decide when it's necessary (vs. what we do now, which is just
 *      rescale every so often).
 *   3. when we're unable to find a pivot above the threshold, could try
 *      rescaling and then repeating the search.
 */

#include "main.h"
#include <malloc.h>
#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "sparse.h"


/* ######################################################################## */
/* sparseCreate */

SparseMatrix_t *sparseCreate()
{
  SparseMatrix_t *sm;
  
  sm = (SparseMatrix_t *) malloc( sizeof(SparseMatrix_t) );
  if (sm != NULL) {
    sm->rows = NULL;
    sm->cols = NULL;
  }

  return sm;
}


/* ######################################################################## */
/* sparseDup */

#ifdef OLDC
SparseMatrix_t *sparseDup(sm)
SparseMatrix_t *sm;
#else
SparseMatrix_t *sparseDup(SparseMatrix_t *sm)
#endif
{
  SparseMatrix_t  *dup;
  RowHeader_t     *rh;
  ElementNode_t   *e;

  dup = sparseCreate();
  if (dup == NULL) {
    error("sparseDup: sparse create failed");
    return NULL;
  }/*then*/

  rh = sm->rows;
  while (rh != NULL) {  /* for each row... */
  
    e = rh->relems;
    while (e != NULL) {  /* for each col in that row... */
      if (Not sparsePut(dup, e->row, e->col, e->value)) {
        error("sparseDup: sparse put failed");
        sparseFree(dup);
        return NULL;
      }/*then*/
      e = e->ncolE;
    }/*while*/

    rh = rh->nrowH;
  }/*while*/

  return dup;
}


/* ######################################################################## */
/* sparseFree */

#ifdef OLDC
int  sparseFree(sm)
SparseMatrix_t *sm;
#else
void sparseFree(SparseMatrix_t *sm)
#endif
{
  RowHeader_t    *rh, *trH;
  ColHeader_t    *ch, *tcH;
  ElementNode_t  *e, *tE;

  if (sm == NULL) 
    return;

  /* first, free col header nodes */
  ch = sm->cols;
  while (ch != NULL) {
    tcH = ch;
    ch = ch->ncolH;
    free(tcH);
  }/*while*/

  /* now go down rows, freeing elements and header nodes */
  rh = sm->rows;
  free(sm);  /* free main header node too! */
  while (rh != NULL) {

    trH = rh;
    e = rh->relems;
    rh = rh->nrowH;
    free(trH);
 
    while (e != NULL) {
      tE = e;
      e = e->ncolE;
      free(tE);
    }/*while*/
  }/*while*/
}


/* ######################################################################## */
/* sparseNumElements */

#ifdef OLDC
long sparseNumElements(sm)
SparseMatrix_t *sm;
#else
long sparseNumElements(SparseMatrix_t *sm)
#endif
{
  long            count;
  RowHeader_t    *rh;
  ElementNode_t  *e;

  rh = sm->rows;
  count = 0;
  while (rh != NULL) {

    e = rh->relems;
    while (e != NULL) {
      count++;
      e = e->ncolE;
    }/*while*/

    rh = rh->nrowH;
  }/*while*/

#ifdef DEBUG  /* make sure that column count yields the same... */
{
  long          colCount;
  ColHeader_t  *ch;

  ch = sm->cols;
  colCount = 0;
  while (ch != NULL) {

    e = ch->celems;
    while (e != NULL) {
      colCount++;
      e = e->nrowE;
    }/*while*/

    ch = ch->ncolH;
  }/*while*/

  if (count != colCount)
    error("row and col counts differ in sparseNumElements!");
}
#endif

  return count;
}


/* ######################################################################## */
/* sparseNumRows */

#ifdef OLDC
Index_t sparseNumRows(sm)
SparseMatrix_t *sm;
#else
Index_t sparseNumRows(SparseMatrix_t *sm)
#endif
{
  Index_t       count;
  RowHeader_t  *rh;

  rh = sm->rows;
  count = 0;
  while (rh != NULL) {
    count++;
    rh = rh->nrowH;
  }

  return count;
}


/* ######################################################################## */
/* sparseMaxRow */

#ifdef OLDC
Index_t sparseMaxRow(sm)
SparseMatrix_t *sm;
#else
Index_t sparseMaxRow(SparseMatrix_t *sm)
#endif
{
  RowHeader_t  *rh;

  rh = getLastRowHPtr(sm);
  if (rh == NULL)
    return 0;

  return rh->row;
}


/* ######################################################################## */
/* sparseNumCols */

#ifdef OLDC
Index_t sparseNumCols(sm)
SparseMatrix_t *sm;
#else
Index_t sparseNumCols(SparseMatrix_t *sm)
#endif
{
  Index_t       count;
  ColHeader_t  *ch;

  ch = sm->cols;
  count = 0;
  while (ch != NULL) {
    count++;
    ch = ch->ncolH;
  }

  return count;
}


/* ######################################################################## */
/* sparseMaxCol */

#ifdef OLDC
Index_t sparseMaxCol(sm)
SparseMatrix_t *sm;
#else
Index_t sparseMaxCol(SparseMatrix_t *sm)
#endif
{
  ColHeader_t  *ch;

  ch = getLastColHPtr(sm);
  if (ch == NULL)
    return 0;

  return ch->col;
}


/* ######################################################################## */
/* sparseGet */
/*
 * Get the desired element and return True; otherwise return 0.0 and
 *   False.
 */
 
#ifdef OLDC
boolean sparseGet(sm, row, col, valuep)
SparseMatrix_t *sm;
Index_t         row;
Index_t         col;
double         *valuep;
#else
boolean sparseGet(SparseMatrix_t *sm, Index_t row, Index_t col, double *valuep)
#endif
{
  RowHeader_t    *rh;
  ElementNode_t  *e;

  assert(row >= 0);
  assert(col >= 0);

  /* search for desired row */
  rh = getThisRowHeader(sm, row);
  if (rh == NULL) {  /* if didn't find row... */ 
    *valuep = 0.0;
    return False;
  }/*then*/

  /* row exists, see if element exists in desired column... */
  e = rh->relems;
  while ( (e != NULL) And (e->col != col) )
    e = e->ncolE;
    
  if (e == NULL) {  /* if didn't find col... */ 
    *valuep = 0.0;
    return False;
  }/*then*/

  /* otherwise we found it! */ 
  *valuep = e->value;
  return True;
}


/* ######################################################################## */
/* sparsePut */ 
/*
 * Puts the desired element into the matrix; if already there, previous
 *   value is overwritten.  If not already there, value is inserted.
 *   Returns True, unless insertion failed (where False is returned).
 */
 
#ifdef OLDC
boolean sparsePut(sm, row, col, value)
SparseMatrix_t *sm; 
Index_t         row;
Index_t         col; 
double          value;
#else
boolean sparsePut(SparseMatrix_t *sm, Index_t row, Index_t col, double value)
#endif
{
  RowHeader_t    *rh, *prevRH, *newRH;
  ColHeader_t    *ch, *prevCH, *newCH;
  ElementNode_t  *e, *prevE, *newE;

  assert(row >= 0);
  assert(col >= 0);

  /* 
   * first, is the element already in the matrix?  If so, overwrite...
   */
   
  /* search for row, then search for the col...   */
  rh = getThisRowHeader(sm, row);
  if (rh != NULL) {  /* found row, does col exist? */

    e = rh->relems;
    while ( (e != NULL) And (e->col != col) )
      e = e->ncolE;
    
    if (e != NULL) {  /* col exists, so replace value and return! */
      e->value = value;
      return True;
    }/*then*/
  }/*then*/

  /* 
   * otherwise element is not in the matrix, so we have to insert it... 
   */

  /* allocate a new element... */
  newE = newElement(row, col, value);
  if (newE == NULL) {
    error("sparsePut: out of memory");
    return False;
  }/*then*/

  /* search for desired row insertion point */ 
  rh = sm->rows;
  prevRH = NULL;
  while ( (rh != NULL) And (rh->row < row) ) {
    prevRH = rh;
    rh = rh->nrowH;
  }/*while*/

  if ( (rh == NULL) Or      /* if no such row, add to end of row list */
       (rh->row > row) ) {  /* or, add between two other rows */ 

    newRH = (RowHeader_t *) malloc( sizeof(RowHeader_t) );
    if (newRH == NULL) {
    error("sparsePut: out of memory");
      return False;
    }/*then*/

    if (prevRH == NULL)  /* link either into header or after prev node */
      sm->rows = newRH;
    else 
      prevRH->nrowH = newRH;

    newRH->nrowH  = rh;
    newRH->prowH  = prevRH;
    newRH->relems = NULL;
    newRH->row    = row;
    
    if (rh == NULL)  /* set prev link of rh (if rh exists!) */
      Nop;
    else rh->prowH = newRH;

    rh = newRH;
  }/*then*/

  /* at this point row exists, now insert in proper column position... */
  insertIntoRow(rh, newE);

  /* finally, we have to go back and link the new element node into */
  /*   the proper column list... (i.e. set its next_row link)       */

  /* search for desired col insertion point */ 
  ch = sm->cols;
  prevCH = NULL;
  while ( (ch != NULL) And (ch->col < col) ) {
    prevCH = ch;
    ch = ch->ncolH;
  }/*while*/

  if ( (ch == NULL) Or      /* if no such col, add to end of row list */
       (ch->col > col) ) {  /* or, add between two other col */ 

    newCH = (ColHeader_t *) malloc( sizeof(ColHeader_t) );
    if (newCH == NULL) {
      error("sparsePut: out of memory");
      return False;
    }/*then*/

    if (prevCH == NULL)  /* link either into header or after prev node */
      sm->cols = newCH;
    else 
      prevCH->ncolH = newCH;

    newCH->ncolH  = ch;
    newCH->pcolH  = prevCH;
    newCH->celems = NULL;
    newCH->col    = col;
    newCH->whichUnknown   = col;
    newCH->colScaleFactor = 1.0;
    
    if (ch == NULL)  /* set prev link of ch (if ch exists!) */
      Nop;
    else ch->pcolH = newCH;

    ch = newCH;
  }/*then*/
  
  /* we have the proper column, now insert new element in proper row order */
  insertIntoColumn(ch, newE);
    
  /* finally, we're done! */
  return True;
}


/* ######################################################################## */
/* sparseScale */ 
/*
 * Scales the matrix by row and then by column, trying to bring matrix to
 *   "equilibration" -- each row and column is adjusted so that largest
 *   (absolute value) element is approximately 1.  Scaling before
 *   factoring usually improves the choice of pivots, and thus accuracy.
 *
 * NOTE: this has implications when you later solve the matrix, so if
 *   you add new elements, you should once again scale, factor, then 
 *   solve.
 */

#ifdef OLDC
boolean sparseScale(sm)
SparseMatrix_t *sm;
#else
boolean sparseScale(SparseMatrix_t *sm)
#endif
{
  return doScaling(sm, False /* not a rescale, first time */);
}
 
#ifdef OLDC
boolean doScaling(sm, rescale)
SparseMatrix_t *sm;
boolean         rescale;
#else
boolean doScaling(SparseMatrix_t *sm, boolean rescale)
#endif
{
  RowHeader_t    *rh;
  ColHeader_t    *ch;
  ElementNode_t  *e;
  double          scaleF, maxval, exp;

  /* 
   * First we scale by row, then by column, though the reverse is
   *   sometimes better (but we can't tell in advance, so we just
   *   choose an order). 
   *
   * When scaling by row, we also update the RHS b, which keeps the
   *   equations correct.  When scaling by column however, this changes
   *   the value for this unknown; thus, we store the column scaling 
   *   factor in the column header node, and use it to correct the
   *   unknowns during the solve process.
   *
   * Scale factors are chosen so that the largest element (absolute 
   *   value) in a given row/column has a magnitude near one.
   */

  /* by row... */
  rh = sm->rows;
  while (rh != NULL) {

    /* first, find largest absolute value... */
    maxval = findMaxInRow(rh);
    if (maxval > 0.0) {  /* no divide by zero! */
    
      /* scale by power of 2 to avoid roundoff during scaling... */
#ifdef THINKC
      scaleF = maxval;
#else

#ifdef SEQUENT
      scaleF = maxval;
#else
      exp = log2(maxval);
      exp = anint(exp);  /* round it up or down, ties up */
      scaleF = exp2(exp);
#endif

#endif

      /* now go back and scale by this value... */
      scaleRowByDivisor(rh, scaleF);
    }/*then*/

    rh = rh->nrowH;  /* next row please */
  }/*while*/

  /* then by column... */
  ch = sm->cols;
  while (ch->ncolH != NULL) {  /* for each column except the RHS b... */

    /* first, find largest absolute value... */
    maxval = findMaxInCol(ch);
    if (maxval == 0.0)  /* no divide by zero! */
      ch->colScaleFactor = 1.0;
    else {
    
      /* scale by power of 2 to avoid roundoff during scaling... */
#ifdef THINKC
      scaleF = maxval;
#else

#ifdef SEQUENT
      scaleF = maxval;
#else
      exp = log2(maxval);
      exp = anint(exp);  /* round it up or down, ties up */
      scaleF = exp2(exp);
#endif

#endif

      /* store scale factor for later use during solve */
      if (rescale == True)  /* on rescaling, factor new scaleF in */
        ch->colScaleFactor = ch->colScaleFactor * scaleF; 
      else 
        ch->colScaleFactor = scaleF;

      /* now go back and scale by this value... */
      scaleColByDivisor(ch, scaleF);
    }/*else*/

    ch = ch->ncolH;  /* next column please */
  }/*while*/
 
  return True;
}


/* ######################################################################## */
/* sparseFactor */ 
/*
 * NOTE: the matrix MUST be scaled before factoring (at least to ensure 
 *   that the column scale factors are properly set).
 */

#ifdef OLDC
boolean sparseFactor(sm)
SparseMatrix_t *sm;
#else
boolean sparseFactor(SparseMatrix_t *sm)
#endif
{
  RowHeader_t    *pivotRH;
  ColHeader_t    *pivotCH;
  ElementNode_t  *e, *pivotE;

  /* can't factor/solve unless square (with RHS b as well) */
  if (sparseNumRows(sm) != (sparseNumCols(sm) - 1)) {
    error("sparseFactor: matrix is not square");
    return False;
  }/*then*/

  /* initially, each column denotes that unknown (swapping changes this) */
  pivotCH = sm->cols; 
  while (pivotCH != NULL) {
    pivotCH->whichUnknown = pivotCH->col;
    pivotCH = pivotCH->ncolH;
  }/*while*/

  pivotCH = sm->cols;  /* we know at least one column since square matrix */
  pivotRH = sm->rows;
  while (pivotCH->ncolH != NULL) {  /* for each column (except RHS b)...  */
   
#ifdef DEBUG
    fprintf(stderr, "factoring column %ld\n", (long) pivotCH->col);
#ifdef VERBOSE 
    { long tot;
      if ((pivotCH->col % 100) == 0) {
        tot = sparseNumElements(sm);
        fprintf(stderr, "matrix currently has %ld elements.\n", tot);
      }
    }
#endif
#endif

    /* (1) compute markowitz products for sub-matrix... */
    computeMarkowitzProducts(sm, pivotRH, pivotCH);

    /* (2) find best pivot element for sub-matrix... */
    pivotE = findBestPivotElement(sm, pivotRH, pivotCH);
    if (pivotE == NULL) {
      error("sparseFactor: no reasonable pivot exists");
      return False;
    }/*then*/

#ifdef VERBOSE 
    fprintf(stderr, "best pivot element in pos (%ld, %ld).\n",
            (long) pivotE->row, (long) pivotE->col);
#endif

    /* (3) move best pivot Element into position */
    pivotE = swapRowsAndCols(sm, pivotRH, pivotCH, pivotE);

#ifdef VERBOSE
    fprintf(stderr, "Rows and columns swapped, best pivot in position.\n");
#endif

    /* (4) scale rows by pivot element to lead element ratio... */
    scaleRowsDueToPivot(pivotE);
    
#ifdef VERBOSE
    fprintf(stderr, "rows have been scaled.\n");
#endif

    /* (5) add fillins -- i.e., each row below pivot row needs to have */
    /*     elements in the same columns as the pivot row...            */
    if ( addFillIns(sm, pivotE) == False ) {
      error("sparseFactor: adding of fillins failed");
      return False;
    }/*then*/

#ifdef VERBOSE
    fprintf(stderr, "Fillins added.\n");
#endif

    /* (6) eliminate pivot col elements, update rest of rows... */
    eliminateColUpdateRows(pivotE);

    /* (7) rescale if looks necessary... */ 
    /* 
     * NOTE: we should really look to see if sm needs to be rescaled.
     *   For now, we just do it every X number of iterations.
     */
#ifdef RescaleDuringFactor
    if ( (pivotCH->col > 0) And ((pivotCH->col % RescaleCount) == 0) )
      if (doScaling(sm, True /* we are rescaling matrix */) == False) {
        error("sparseFactor: rescaling of matrix failed");
        return False;
      }/*then*/
#endif

    /* (8) and repeat... */
    pivotCH = pivotCH->ncolH;
    pivotRH = pivotRH->nrowH;
  }/*while*/

  return True;
}


/* ######################################################################## */
/* sparseSolve */ 
/*
 * NOTE: since solving Ax = b, this means x must be an array with at
 *   least MaxCol entries (only need MaxCol entries, and not MaxCol+1, 
 *   since last col is assumed to be the RHS b, which we can ignore in 
 *   this case).
 */

#ifdef OLDC
boolean sparseSolve(sm, x)
SparseMatrix_t *sm;
double          x[];
#else
boolean sparseSolve(SparseMatrix_t *sm, double x[])
#endif
{
  Index_t         col, unk, *mapUnk;
  double          divisor;
  RowHeader_t    *rh;
  ColHeader_t    *ch;
  ElementNode_t  *e, *bi, *ci;
  
  /* can't factor/solve unless square (with RHS b as well) */
  if (sparseNumRows(sm) != (sparseNumCols(sm) - 1)) {
    error("sparseSolve: matrix is not square");
    return False;
  }/*then*/

  /*
   * We have to map from columns in matrix to the actual unknowns
   *   they represent (which is denoted by the columns "which unknown"
   *   field...
   */

  mapUnk = (Index_t *) malloc( sizeof(Index_t) * sparseMaxCol(sm) );
  if (mapUnk == NULL) {
    error("sparseSolve: out of memory");
    return False;
  }/*then*/

  ch = sm->cols;
  while (ch->ncolH != NULL) {  /* for every unknown except RHS b... */
    mapUnk[ch->col] = ch->whichUnknown;
    ch = ch->ncolH;
  }/*while*/

  /* init solution vector to zero... */
  col = sparseMaxCol(sm);
  col--;  /* we don't solve last row, which is RHS b */
  while (col >= 0) {
    x[col] = 0.0;
    col--;
  }/*while*/
  
  /* 
   * We solve matrix by starting with last row (which has only one
   *   unknown), solving this, and then moving up to the prev row,
   *   and so on...
   */
   
  /* first, find bottom-most, right-most element -- last elem of RHS b */
  rh = getLastRowHPtr(sm);
  assert( rh != NULL );
    
  e = rh->relems;
  assert( e != NULL );
  while (e->ncolE != NULL)
    e = e->ncolE;
  
  bi = e;
  ci = e;  /* to get us started... */
  
  do {
 
    /*
     * bi is the RHS element of equation to solve.  We have solved all bi's 
     *   that come after in RHS vector.  So, find first unknown in bi's row,
     *   simplify so that constant for this var is 1, and then solve for 
     *   var...
     */
    
    e = bi;  /* given previous ci, next unknown is first unsolved column */
    while ((e != NULL) And (e->col >= ci->col))
      e = e->pcolE;
      
    if (e == NULL) {
      error("sparseSolve: could not locate next unknown element");
      free(mapUnk);
      return False;
    }/*then*/
  
    ci = e;  /* this is new ci -- last column not yet solved in row */
    
    /* simplify row by dividing through with ci's value */
    divisor = ci->value;
    if (divisor == 0.0) {
      error("sparseSolve: divisor is zero");
      free(mapUnk);
      return False;
    }/*then*/
    
    ci->value = 1.0;
    e = ci->ncolE;
    while (e != NULL) {  /* update remaining elements in row, including bi */
      e->value = e->value / divisor;
      e = e->ncolE;
    }/*while*/
    
    /* now solve for unknown */
    unk = mapUnk[ci->col];
    x[unk] = bi->value;  /* bi is RHS b for this row -- initial val */
    e = ci;
    e = e->ncolE;
    while (e != bi) {
      x[unk] = x[unk] - (e->value * x[ mapUnk[e->col] ]);
      e = e->ncolE;
    }/*while*/
    
    /* repeat for previous row, which now has only 1 unknown... */
    bi = bi->prowE;  /* move up one row in RHS b... */
  
  } while (bi != NULL);  /* repeat until RHS b is empty... */

  /* 
   * Finally, correct solution by column scale factors... 
   */
  
  ch = sm->cols;  /* we know ch is non-NULL, since at least one row */
  while (ch->ncolH != NULL) {  /* for all unknowns (but not RHS b)  */
    unk = mapUnk[ch->col];
    x[unk] = x[unk] / ch->colScaleFactor;
    ch = ch->ncolH;
  }/*while*/
  
  /* if got this far, success! */
  free(mapUnk);

  return True;
}


/* ######################################################################## */
/* sparsePrint */ 
 
#ifdef OLDC
int  sparsePrint(sm)
SparseMatrix_t *sm;
#else
void sparsePrint(SparseMatrix_t *sm)
#endif
{
  RowHeader_t    *rh;
  ElementNode_t  *e;

  printf("\n** Dump of sparse matrix **\n");

  printf("\nRow-major order (%ld rows, max row = %ld, # elements %ld):\n", 
          (long) sparseNumRows(sm), (long) sparseMaxRow(sm), sparseNumElements(sm));

  rh = sm->rows;
  while (rh != NULL) {

    printf(" Row %ld:\n", (long) rh->row);

    e = rh->relems;
    while (e != NULL) {
      printf("   (%ld,%ld) = %e\n", (long) e->row, (long) e->col, e->value);
      e = e->ncolE;
    }/*while*/

    rh = rh->nrowH;
  }/*while*/

#ifdef DEBUG  /* then print in column-major order as well */
  {
    ColHeader_t  *ch;
    
    printf("\nColumn-major order (%ld cols, max col = %ld, # elements %ld):\n",
            (long) sparseNumCols(sm), (long) sparseMaxCol(sm), sparseNumElements(sm));


    ch = sm->cols;
    while (ch != NULL) {

      printf(" Col %ld:\n", (long) ch->col);

      e = ch->celems;
      while (e != NULL) {
        printf("   (%ld,%ld) = %e\n", (long) e->row, (long) e->col, e->value);
        e = e->nrowE;
      }/*while*/

      ch = ch->ncolH;
    }/*while*/
  }
#endif

  printf("\n** Done **\n");
  fflush(stdout);
}


/* ######################################################################## */
/* sparseCheckInvariants */
/*
 * Checks various properties that should always hold for a well-formed 
 *   sparse matrix.  If a property is found to be invalid, this routine
 *   will never return; instead it dies with an error message.
 *
 * NOTE: this is used to debug under the assumption that there are
 *   rows 0 .. maxRow, and columns 0 .. maxCol, i.e. consecutive rows
 *   and columns.  Also, maxRow and maxCol denote the max row and column 
 *   numbers you specified when you built the matrix, where maxCol 
 *   includes the RHS.
 */

#ifdef DEBUG

#ifdef OLDC
int sparseCheckInvariants(sm, maxRow, maxCol)
SparseMatrix_t  *sm;
long             maxRow;
long             maxCol;
#else
void sparseCheckInvariants(SparseMatrix_t *sm, long maxRow, long maxCol)
#endif
{
  RowHeader_t    *rh, *prevRH;
  ColHeader_t    *ch, *prevCH;
  ElementNode_t  *e,  *prevE;
  Index_t         r, c;

  rh = sm->rows;
  ch = sm->cols; 
  prevRH = NULL;
  prevCH = NULL;

  r = 0;
  while (rh != NULL) {
    if (rh->prowH != prevRH)
      error("sparseCheckInvariants: prev rh ptr is incorrect");
    if (rh->row != r) 
      error("sparseCheckInvariants: rh row number is incorrect");

    e = rh->relems;
    prevE = NULL;
    while (e != NULL) {
      if (e->pcolE != prevE)
        error("sparseCheckInvariants: prev col elem ptr is incorrect");
      if (e->row != r) 
        error("sparseCheckInvariants: elem row number is incorrect");

      prevE = e;
      e = e->ncolE;
    }/*while*/

    r++;
    prevRH = rh;
    rh = rh->nrowH;
  }/*while*/

  if (r != (maxRow + 1))
    error("sparseCheckInvariants: number of rows is incorrect"); 

  c = 0;
  while (ch != NULL) {
    if (ch->pcolH != prevCH)
      error("sparseCheckInvariants: prev ch ptr is incorrect");
    if (ch->col != c) 
      error("sparseCheckInvariants: ch col number is incorrect");

    e = ch->celems;
    prevE = NULL;
    while (e != NULL) {
      if (e->prowE != prevE)
        error("sparseCheckInvariants: prev row elem ptr is incorrect");
      if (e->col != c) 
        error("sparseCheckInvariants: elem col number is incorrect");

      prevE = e;
      e = e->nrowE;
    }/*while*/

    c++;
    prevCH = ch;
    ch = ch->ncolH;
  }/*while*/

  if (c != (maxCol + 1)) 
    error("sparseCheckInvariants: number of cols is incorrect"); 

  /* special check: the RHS should have one element for every row! */
  e = prevCH->celems;  /* RHS should be last row in matrix...      */
  r = 0; 
  while (e != NULL) {
    if (e->row != r)
      error("sparseCheckInvariants: RHS is missing a row"); 
    r++;
    e = e->nrowE;
  }/*while*/
  if (r != (maxRow + 1)) 
    error("sparseCheckInvariants: RHS has too few or too many rows");

  /* all is well if we get to here! */
}

#endif


/* ##############################  LOCAL  ################################# */

/* ######################################################################## */
#ifdef OLDC
RowHeader_t  *getLastRowHPtr(sm)
SparseMatrix_t *sm;
#else
RowHeader_t  *getLastRowHPtr(SparseMatrix_t *sm)
#endif
{
  RowHeader_t  *rh;
  
  rh = sm->rows;
  if (rh == NULL)
    return NULL;
    
  while (rh->nrowH != NULL)
    rh = rh->nrowH;
  
  return rh;
}

/* ######################################################################## */
#ifdef OLDC
ColHeader_t  *getLastColHPtr(sm)
SparseMatrix_t *sm;
#else
ColHeader_t  *getLastColHPtr(SparseMatrix_t *sm)
#endif
{
  ColHeader_t  *ch;
  
  ch = sm->cols;
  if (ch == NULL)
    return NULL;
    
  while (ch->ncolH != NULL)
    ch = ch->ncolH;
  
  return ch;
}

/* ######################################################################## */
#ifdef OLDC
RowHeader_t  *getThisRowHeader(sm, row)
SparseMatrix_t *sm;
Index_t         row;
#else
RowHeader_t  *getThisRowHeader(SparseMatrix_t *sm, Index_t row)
#endif
{
  RowHeader_t  *rh;
  
  rh = sm->rows;
  while ((rh != NULL) And (rh->row != row))
    rh = rh->nrowH;
  
  return rh;
}

/* ######################################################################## */
#ifdef OLDC
ColHeader_t  *getThisColHeader(sm, col)
SparseMatrix_t *sm; 
Index_t         col;
#else
ColHeader_t  *getThisColHeader(SparseMatrix_t *sm, Index_t col)
#endif
{
  ColHeader_t  *ch;
  
  ch = sm->cols;
  while ((ch != NULL) And (ch->col != col))
    ch = ch->ncolH;
  
  return ch;
}

/* ######################################################################## */
#ifdef OLDC
int  computeMarkowitzProducts(sm, pivotRH, pivotCH)
SparseMatrix_t *sm;
RowHeader_t    *pivotRH;
ColHeader_t    *pivotCH;
#else
void computeMarkowitzProducts(SparseMatrix_t *sm, 
                              RowHeader_t    *pivotRH, 
                              ColHeader_t    *pivotCH)
#endif
{
  RowHeader_t    *rh;
  ColHeader_t    *ch;
  ElementNode_t  *e, *startE;
  Index_t         count;
  
  /* Def: markowitz product for an element is: 
   *         # elements in its row - 1  *  # elements in its col - 1
   */

  /* first, find the number of elems in each row... */
  rh = pivotRH;
  while (rh != NULL) {

    e = rh->relems;  /* find start of sub-matrix */
    while ((e != NULL) And (e->col < pivotCH->col))
      e = e->ncolE;

    assert( e != NULL );        /* due to RHS b... */
    startE = e;

    /* compute # of elems in the row... */
    count = 0;
    while (e != NULL) {
      count++;
      e = e->ncolE;
    }/*while*/

    /* ... and store first part of MP computation in each elem */
    e = startE;
    while (e->ncolE != NULL) {  /* for all except RHS b... */
      e->markowitzProduct = count - 1;
      e = e->ncolE;
    }/*while*/

    rh = rh->nrowH;
  }/*while*/
  
  /* now compute number in cols and finish MP calculations... */
  ch = pivotCH;
  assert( ch != NULL );
  while (ch->ncolH != NULL) {  /* for all cols except RHS b... */

    e = ch->celems;
    while ((e != NULL) And (e->row < pivotRH->row)) 
      e = e->nrowE;
      
    assert( e != NULL );        /* due to RHS b... */
    startE = e;

    /* compute # of elems in the col... */
    count = 0;
    while (e != NULL) {
      count++;
      e = e->nrowE;
    }/*while*/

    /* ... and compute final MP value */
    e = startE;
    while (e != NULL) {
      e->markowitzProduct = e->markowitzProduct * (count - 1);
      e = e->nrowE;
    }/*while*/

    ch = ch->ncolH;
  }/*while*/
}

/* ######################################################################## */
#ifdef OLDC
ElementNode_t *findBestPivotElement(sm, pivotRH, pivotCH)
SparseMatrix_t *sm; 
RowHeader_t    *pivotRH;
ColHeader_t    *pivotCH;
#else
ElementNode_t *findBestPivotElement(SparseMatrix_t *sm, 
                                    RowHeader_t    *pivotRH,
                                    ColHeader_t    *pivotCH)
#endif
{
  long            minMP, minRow, maxMP;
  RowHeader_t    *rh;
  ElementNode_t  *e, *pivotE;
  int             i;

  assert( pivotRH != NULL);  /* there should be a pivot row */
  assert( pivotCH != NULL);  /* there should be a pivot col */

  minMP = MaxLong;
  pivotE = NULL;

  /* we consider every element in the sub-matrix, except RHS b... */
  rh = pivotRH;
  while (rh != NULL) {
    
    e = rh->relems;
    while ((e != NULL) And (e->col < pivotCH->col))
      e = e->ncolE;
      
    /* e is start of sub-matrix for this row... */
    assert( e != NULL );  /* since CH will never be the RHS b */
    while (e->ncolE != NULL) {  /* for all except RHS b... */

      if ((e->markowitzProduct < minMP) And 
          (fabs(e->value) >= PivotThreshold)) {
        pivotE = e;
        minMP  = e->markowitzProduct;
      }/*then*/

      e = e->ncolE;
    }/*while*/ 
    
    rh = rh->nrowH;
  }/*while*/

  if (pivotE != NULL)  /* if found a pivot, return now! */
    return pivotE;

  /* 
   * NOTE: we assume above doesn't fail very often, so we don't bother
   *   to parallelize the code that follows below.
   */

  /* Didn't find a pivot first time through; try again, only this time */
  /*   we'll ignore the threshold...                                   */
#ifdef DEBUG
  fprintf(stderr, "findBestPivotElement: no pivots above threshold, repeating search.\n");
#endif

  /* NOTE: rescale matrix? */

  minMP = MaxLong;
  pivotE = NULL;
  
  /* we consider every element in the sub-matrix, except RHS b... */
  rh = pivotRH;
  while (rh != NULL) {
    
    e = rh->relems;
    while ((e != NULL) And (e->col < pivotCH->col))
      e = e->ncolE;
      
    /* e is start of sub-matrix for this row... */
    assert( e != NULL );  /* since CH will never be the RHS b */
    while (e->ncolE != NULL) {  /* for all except RHS b... */

      if ((e->markowitzProduct < minMP) And (e->value != 0.0)) {
        pivotE = e;
        minMP  = e->markowitzProduct;
      }/*then*/
      e = e->ncolE;
    }/*while*/ 
    
    rh = rh->nrowH;
  }/*while*/

  if (pivotE != NULL)  /* if found a pivot, return now! */
    return pivotE;

  /* all possible pivots are zero, nothing we can do... */
  error("findBestPivotElement: all possible pivots are zero, must fail"); 

  return NULL; 
}

/* ######################################################################## */
#ifdef OLDC
int  unlinkFromRow(rh, e)
RowHeader_t   *rh;
ElementNode_t *e;
#else
void unlinkFromRow(RowHeader_t *rh, ElementNode_t *e)
#endif
{
  if (e->pcolE == NULL)      /* if first in list, update header */
    rh->relems = e->ncolE;
  else e->pcolE->ncolE = e->ncolE;  /* else update prev node's next field */
  
  if (e->ncolE == NULL)      /* if last in list, no next to update... */
    Nop ;
  else e->ncolE->pcolE = e->pcolE;  /* else update next node's prev field */
}                                 

/* ######################################################################## */
#ifdef OLDC
int  unlinkFromColumn(ch, e)
ColHeader_t   *ch;
ElementNode_t *e;
#else
void unlinkFromColumn(ColHeader_t *ch, ElementNode_t *e)
#endif
{
  if (e->prowE == NULL)      /* if first in list, update header */
    ch->celems = e->nrowE;
  else e->prowE->nrowE = e->nrowE;  /* else update prev node's next field */
  
  if (e->nrowE == NULL)      /* if last in list, no next to update... */
    Nop ;
  else e->nrowE->prowE = e->prowE;  /* else update next node's prev field */
}                                 

/* ######################################################################## */
#ifdef OLDC
int  insertIntoColumn(ch, newE)
ColHeader_t   *ch;
ElementNode_t *newE;
#else
void insertIntoColumn(ColHeader_t *ch, ElementNode_t *newE)
#endif
{
  ElementNode_t  *e, *prevE;
  
  e = ch->celems;
  prevE = NULL;
  while ( (e != NULL) And (e->row < newE->row) ) {
    prevE = e;
    e = e->nrowE;
  }/*while*/

  if (prevE == NULL)  /* link either into header or after prev node */
    ch->celems = newE;
  else 
    prevE->nrowE = newE;

  newE->nrowE = e;     /* new element is between prev and e... */
  newE->prowE = prevE; 
  
  if (e == NULL)  /* set prev link of e as well (if e exists!) */
    Nop ;
  else e->prowE = newE;
}                                 

/* ######################################################################## */
#ifdef OLDC
int  linkIntoColumnBetween(ch, prevE, curE, newE)
ColHeader_t   *ch;
ElementNode_t *prevE;
ElementNode_t *curE;
ElementNode_t *newE;
#else
void linkIntoColumnBetween(ColHeader_t *ch, ElementNode_t *prevE,
                                  ElementNode_t *curE, ElementNode_t *newE)
#endif
{  
  if (prevE == NULL)  /* link either into header or after prev node */
    ch->celems = newE;
  else 
    prevE->nrowE = newE;

  newE->nrowE = curE;  /* new element is between prev and e... */
  newE->prowE = prevE; 
  
  if (curE == NULL)  /* set prev link of e as well (if e exists!) */
    Nop ;
  else curE->prowE = newE;
}

/* ######################################################################## */
#ifdef OLDC
int  insertIntoRow(rh, newE)
RowHeader_t   *rh;
ElementNode_t *newE;
#else
void insertIntoRow(RowHeader_t *rh, ElementNode_t *newE)
#endif
{
  ElementNode_t  *e, *prevE;
  
  e = rh->relems;
  prevE = NULL;
  while ( (e != NULL) And (e->col < newE->col) ) {
    prevE = e;
    e = e->ncolE;
  }/*while*/
  
  if (prevE == NULL)  /* link either into header or after prev node */
    rh->relems = newE;
  else 
    prevE->ncolE = newE;

  newE->ncolE = e;     /* new element is between prev and e... */
  newE->pcolE = prevE; 

  if (e == NULL)  /* set prev link of e as well (if e exists!) */
    Nop ;
  else e->pcolE = newE;
}

/* ######################################################################## */
#ifdef OLDC
int  linkIntoRowBetween(rh, prevE, curE, newE)
RowHeader_t   *rh;
ElementNode_t *prevE;
ElementNode_t *curE;
ElementNode_t *newE;
#else
void linkIntoRowBetween(RowHeader_t *rh, ElementNode_t *prevE,
                               ElementNode_t *curE, ElementNode_t *newE)
#endif
{  
  if (prevE == NULL)  /* link either into header or after prev node */
    rh->relems = newE;
  else 
    prevE->ncolE = newE;

  newE->ncolE = curE;     /* new element is between prev and e... */
  newE->pcolE = prevE; 

  if (curE == NULL)  /* set prev link of e as well (if e exists!) */
    Nop ;
  else curE->pcolE = newE;
}

/* ######################################################################## */
#ifdef OLDC
ElementNode_t *swapRowsAndCols(sm, pivotRH, pivotCH, realPivotE)
SparseMatrix_t *sm;
RowHeader_t    *pivotRH;
ColHeader_t    *pivotCH;
ElementNode_t  *realPivotE;
#else
ElementNode_t *swapRowsAndCols(SparseMatrix_t *sm, RowHeader_t *pivotRH,
                        ColHeader_t *pivotCH, ElementNode_t *realPivotE)
#endif
{
  ColHeader_t    *ch, *leftCH, *rightCH;
  RowHeader_t    *rh, *topRH, *botRH;
  ElementNode_t  *leftE, *rightE, *prevE, *curE;
  ElementNode_t  *nextLeftE, *prevRightE, *nextRightE, *prevLeftE;
  ElementNode_t  *topE, *botE;
  ElementNode_t  *nextTopE, *prevBotE, *nextBotE, *prevTopE;
  double          tval;
  Index_t         tunk;

  /* (1) do we need to swap columns? */
  if (realPivotE->col != pivotCH->col) {

    /* if unequal, real pivot column is to right of pivot column... */
    leftCH = pivotCH;
    rightCH = leftCH->ncolH;
    while (rightCH->col != realPivotE->col)
      rightCH = rightCH->ncolH;

    /* march down left column, moving elems to right column */
    leftE = leftCH->celems;
    prevRightE = NULL;
    rightE = rightCH->celems;
    rh = sm->rows; 
    while (leftE != NULL) {

      while (rh->row != leftE->row)
        rh = rh->nrowH;

      while ((rightE != NULL) And (rightE->row < leftE->row)) {
        prevRightE = rightE;
        rightE = rightE->nrowE;
      }
      
      /* got left elem to move, and dest in right column... */
      if ((rightE != NULL) And 
          (leftE->row == rightE->row)) {  /* then just swap values */
        tval = leftE->value;
        leftE->value = rightE->value;
        rightE->value = tval;
        rightE->col = leftE->col;  /* mark as swapped so don't swap back */
        /* keep track of real pivot! */
        if (realPivotE == rightE)
          realPivotE = leftE; 

        /* move to next elem in left column, and repeat... */
        leftE = leftE->nrowE; 
      }
      else {  /* put left elem in right column where currently no elem */

        /* first we move left elem within row... */
        prevE = leftE;
        curE = leftE->ncolE;
        while ((curE != NULL) And (curE->col < rightCH->col)) {
          prevE = curE;
          curE = curE->ncolE;
        } 
   
        /* insert between prevE and curE... */
        if (prevE == leftE)  /* then don't move it! */
          ;
        else {
          unlinkFromRow(rh, leftE);
          linkIntoRowBetween(rh, prevE, curE, leftE);
        }/*else*/

        /* now insert into it's new column... */
        nextLeftE = leftE->nrowE;  /* save before we change it! */
        unlinkFromColumn(leftCH, leftE);
        linkIntoColumnBetween(rightCH, prevRightE, rightE, leftE);
        /* NOTE: we're update col value in leftE later. */
   
        /* next column insertion has to follow after leftE... */
        prevRightE = leftE;
        /* move to next elem in left column, and repeat... */
        leftE = nextLeftE;
      }/*else*/
    }/*while*/

    /*
     * At this point, all values in left column have been moved to
     *   the right column.  Now, we go down the right column and 
     *   move any elems with this column number to the left column.
     */

    rightE = rightCH->celems;
    prevLeftE = NULL;
    leftE = leftCH->celems;
    rh = sm->rows; 
    while (rightE != NULL) {

      while (rh->row != rightE->row)
        rh = rh->nrowH;

      if (rightE->col == rightCH->col) {  /* then move to left col */

        while ((leftE != NULL) And (leftE->row < rightE->row)) {
          prevLeftE = leftE;
          leftE = leftE->nrowE;
        }
      
        /* first we move right elem within row... */
        prevE = rightE->pcolE;  /* we go left in row! */
        curE = rightE;
        while ((prevE != NULL) And (prevE->col > leftCH->col)) {
          curE = prevE;
          prevE = prevE->pcolE;
        } 
   
        /* insert between prevE and curE... */
        if (curE == rightE)  /* then don't move it! */
          ;
        else {
          unlinkFromRow(rh, rightE);
          linkIntoRowBetween(rh, prevE, curE, rightE);
        }/*else*/

        /* now insert into it's new column... */
        nextRightE = rightE->nrowE;  /* save before we change it! */
        unlinkFromColumn(rightCH, rightE);
        linkIntoColumnBetween(leftCH, prevLeftE, leftE, rightE);
        rightE->col = leftCH->col;
   
        /* next column insertion has to follow after rightE... */
        prevLeftE = rightE;

        rightE = nextRightE;  /* and repeat... */
      }/*then*/
      else 
        rightE = rightE->nrowE;
    }/*while*/

    /* 
     * Finally, swap scale factors in col headers, unknowns in col
     *   headers, and fix column values in elems in right column...
     */

    tval = leftCH->colScaleFactor;
    leftCH->colScaleFactor = rightCH->colScaleFactor;
    rightCH->colScaleFactor = tval;

    tunk = leftCH->whichUnknown;
    leftCH->whichUnknown = rightCH->whichUnknown;
    rightCH->whichUnknown = tunk;

    rightE = rightCH->celems;
    while (rightE != NULL) {
      rightE->col = rightCH->col;
      rightE = rightE->nrowE;
    }/*while*/

  }/*then*/
  

  /* (2) do we need to swap rows? */
  if (realPivotE->row != pivotRH->row) {

    /*
     * same idea as above: move top row down to bottom, then bottom
     *   up to top...
     */

    /* if unequal, real pivot row is below pivot row... */
    topRH = pivotRH;
    botRH = topRH->nrowH;
    while (botRH->row != realPivotE->row)
      botRH = botRH->nrowH;

    /* march along top row, moving elems to bottom row */
    topE = topRH->relems;
    prevBotE = NULL;
    botE = botRH->relems;
    ch = sm->cols;
    while (topE != NULL) {

      while (ch->col != topE->col)  /* update ch to follow us */
        ch = ch->ncolH;

      while ((botE != NULL) And (botE->col < topE->col)) {
        prevBotE = botE;
        botE = botE->ncolE;
      }
      
      /* got top elem to move, and dest in bottom row... */
      if ((botE != NULL) And 
          (topE->col == botE->col)) {  /* then just swap values */
        tval = topE->value;
        topE->value = botE->value;
        botE->value = tval;
        botE->row = topE->row;  /* mark as swapped so don't swap back */
        /* keep track of real pivot! */
        if (realPivotE == botE)
          realPivotE = topE; 

        /* move to next elem in top row, and repeat... */
        topE = topE->ncolE;
      }
      else {  /* put top elem in bottom row where currently no elem */

        /* first we move top elem within column... */
        prevE = topE;
        curE = topE->nrowE;
        while ((curE != NULL) And (curE->row < botRH->row)) {
          prevE = curE;
          curE = curE->nrowE;
        } 
   
        /* insert between prevE and curE... */
        if (prevE == topE)  /* then don't move it! */
          ;
        else {
          unlinkFromColumn(ch, topE);
          linkIntoColumnBetween(ch, prevE, curE, topE);
        }/*else*/

        /* now insert into it's new row... */
        nextTopE = topE->ncolE;  /* save before we change it! */
        unlinkFromRow(topRH, topE);
        linkIntoRowBetween(botRH, prevBotE, botE, topE);
        /* NOTE: we're update row value in topE later. */
   
        /* next row insertion has to follow after topE... */
        prevBotE = topE;
        /* move to next elem in top row, and repeat... */
        topE = nextTopE;
      }/*else*/
    }/*while*/

    /*
     * At this point, all values in top row have been moved to
     *   the bottom row.  Now, we go along the bottom row and 
     *   move any elems with this row number to the top row.
     */

    botE = botRH->relems;
    prevTopE = NULL;
    topE = topRH->relems;
    ch = sm->cols; 
    while (botE != NULL) {

      while (ch->col != botE->col)  /* update ch to follow us */
        ch = ch->ncolH;

      if (botE->row == botRH->row) {  /* then move to top row */

        while ((topE != NULL) And (topE->col < botE->col)) {
          prevTopE = topE;
          topE = topE->ncolE;
        }
      
        /* first we move bot elem within column... */
        prevE = botE->prowE;  /* we are moving up! */
        curE = botE;
        while ((prevE != NULL) And (prevE->row > topRH->row)) {
          curE = prevE;
          prevE = prevE->prowE;
        } 
   
        /* insert between prevE and curE... */
        if (curE == botE)  /* then don't move it! */
          ;
        else {
          unlinkFromColumn(ch, botE);
          linkIntoColumnBetween(ch, prevE, curE, botE);
        }/*else*/

        /* now insert into it's new row... */
        nextBotE = botE->ncolE;  /* save before we change it! */
        unlinkFromRow(botRH, botE);
        linkIntoRowBetween(topRH, prevTopE, topE, botE);
        botE->row = topRH->row;
   
        /* next row insertion has to follow after botE... */
        prevTopE = botE;
        /* move to next elem in top row, and repeat... */
        botE = nextBotE;
      }/*then*/
      else
        botE = botE->ncolE;
    }/*while*/
    
    /* 
     * Finally, fix row values in elems in bottom row...
     */

    botE = botRH->relems;
    while (botE != NULL) {
      botE->row = botRH->row;
      botE = botE->ncolE;
    }/*while*/

  }/*then*/

  return realPivotE;
}

/* ######################################################################## */
#ifdef OLDC
int  scaleRowsDueToPivot(pivotE)
ElementNode_t *pivotE;
#else
void scaleRowsDueToPivot(ElementNode_t *pivotE)
#endif
{
  double          scaleF;
  ElementNode_t  *startE, *e;

  startE = pivotE->nrowE;    /* scale the rows that follow...  */

  while (startE != NULL) {
    
    if (startE->value != 0.0) {  /* no divide by zero! */
    
      scaleF = pivotE->value / startE->value;
      e = startE->ncolE;   /* skip first, the rest get scaled... */
      while (e != NULL) {  /* for rest of elements in the row... */
        e->value = e->value * scaleF;
        e = e->ncolE;
      }/*while*/
      
    }/*then*/
    
    startE = startE->nrowE;
  }/*while*/
}

/* ######################################################################## */
#ifdef OLDC
boolean addFillIns(sm, pivotE)
SparseMatrix_t *sm; 
ElementNode_t  *pivotE;
#else
boolean addFillIns(SparseMatrix_t *sm, ElementNode_t *pivotE)
#endif
{
  ElementNode_t  *startRE, *otherPRE, *prevE, *curE, *newE;
  ElementNode_t  *fillinList, *aFillin, *pFillin;
  int             i;

  /* first, go down each column in pivot row, adding necessary column  */
  /*   elements to rows below pivot row that we will eliminate...      */
  /* 
   * NOTE: we just insert in column, we'll add row links later.
   */ 

  otherPRE = pivotE->ncolE;  /* first column to process */
  fillinList = NULL;
  while (otherPRE->ncolE != NULL) {  /* process all cols except RHS b */

    startRE = pivotE->nrowE;   /* first row to process */
    prevE = otherPRE;       
    curE  = otherPRE->nrowE;
    pFillin = NULL;            
    aFillin = fillinList;      /* reset so we can insert in order */ 
    while (startRE != NULL) {  /* all rows */

      /* search col for place to add new element */
      while ((curE != NULL) And (curE->row < startRE->row)) {
        prevE = curE;
        curE  = curE->nrowE;
      }/*while*/

      if ((curE == NULL) Or 
          (curE->row != startRE->row)) {  /* then add between prev and cur */

        newE = newElement(startRE->row, otherPRE->col, 0.0);
        if (newE == NULL) {
          error("addFillIns: out of memory");
          return False;
        }/*then*/

        /* insert into column (don't need column header, since we know */
        /*   prevE will not be NULL).                                  */
        linkIntoColumnBetween(NULL, prevE, curE, newE); 
        prevE = newE;  /* new elem is now prev to cur */

        /* add to fill in list so we can process fillins easily... */
        while (aFillin != NULL) {
          if (newE->row > aFillin->row)  /* keep list in row order... */
            pFillin = aFillin, aFillin = aFillin->ncolE;
          else if (newE->row == aFillin->row)  /* then insert by column... */
            if (newE->col > aFillin->col)
              pFillin = aFillin, aFillin = aFillin->ncolE;
            else break;
          else break;  /* we found the insertion spot */
        }/*while*/
        newE->ncolE = aFillin;
        if (pFillin == NULL)
          fillinList = newE;
        else pFillin->ncolE = newE;
        aFillin = newE;  /* newE now follows prev fillin... */

      }/*then*/

      startRE = startRE->nrowE;
    }/*while*/

    otherPRE = otherPRE->ncolE;
  }/*while*/

  /* now we go along each row and add the new column elems to it... */

  startRE = pivotE->nrowE;      /* first row to process */
  while (fillinList != NULL) {  /* for each fillin to be added... */

    while (startRE->row != fillinList->row)  /* find start of its row */
      startRE = startRE->nrowE;

    prevE = startRE;  /* start at head of row to find insertion point(s) */
    curE  = startRE->ncolE;
    while ((fillinList != NULL) And  /* for each fillin in this row... */
           (startRE->row == fillinList->row)) { 

      /* find insertion point */
      while ((curE != NULL) And (curE->col < fillinList->col))
        prevE = curE, curE  = curE->ncolE;

      /* insert into row (don't need row header, since we know prevE <> NULL) */
      newE = fillinList;  /* save next node in fillin list before insertion!! */
      fillinList = fillinList->ncolE;     
      linkIntoRowBetween(NULL, prevE, curE, newE); 
      prevE = newE;  /* new elem is now prev to cur */
    }/*while*/

  }/*while*/

  return True;
}

/* ######################################################################## */
#ifdef OLDC
int  eliminateColUpdateRows(pivotE)
ElementNode_t *pivotE;
#else
void eliminateColUpdateRows(ElementNode_t *pivotE)
#endif
{
  ElementNode_t  *startE, *otherPRE, *e, *te;

  /* first, eliminate elements in same column as pivot element... */
  e = pivotE->nrowE;
  while (e != NULL) {
    e->value = 0.0;
    e = e->nrowE;
  }/*while*/

  /* now, update other columns by subtracting pivot row from each row... */
  otherPRE = pivotE->ncolE;   /* elements after pivot element...*/
  while (otherPRE != NULL) {  /* for each column including the RHS b...  */

    startE = pivotE->nrowE;   /* first row to update in this column */
    e = otherPRE->nrowE;      /* first potential column element...  */
    while (startE != NULL) {  /* do all the rows and this column... */
    
      while (e->row != startE->row)  /* find col element for this row */
        e = e->nrowE;  /* NOTE: we know exists since did fillins!     */
    
      /* found element in proper row and column, update its value...  */
      e->value = e->value - otherPRE->value;

      startE = startE->nrowE;  /* next row to update? */
    }/*while*/

    otherPRE = otherPRE->ncolE;  /* next pivot row element? */
  }/*while*/
}

/* ######################################################################## */
#ifdef OLDC
double findMaxInRow(rh)
RowHeader_t *rh;
#else
double findMaxInRow(RowHeader_t *rh)
#endif
{
  double          maxval;
  ElementNode_t  *e;

  maxval = 0.0;
  e = rh->relems;
  while (e != NULL) {  
    if (fabs(e->value) > maxval)
      maxval = fabs(e->value);
    e = e->ncolE;
  }/*while*/
 
  return maxval;
}

/* ######################################################################## */
#ifdef OLDC
double findMaxInCol(ch)
ColHeader_t *ch;
#else
double findMaxInCol(ColHeader_t *ch)
#endif
{
  double          maxval;
  ElementNode_t  *e;

  maxval = 0.0;
  e = ch->celems;
  while (e != NULL) {  
    if (fabs(e->value) > maxval)
      maxval = fabs(e->value);
    e = e->nrowE;
  }/*while*/

  return maxval;
}

/* ######################################################################## */
#ifdef OLDC
int  scaleRowByDivisor(rh, scaleF)
RowHeader_t *rh;
double       scaleF;
#else
void scaleRowByDivisor(RowHeader_t *rh, double scaleF)
#endif
{
  ElementNode_t  *e;

  e = rh->relems;
  while (e != NULL) {  
    e->value = e->value / scaleF;
    e = e->ncolE;
  }/*while*/
}

/* ######################################################################## */
#ifdef OLDC
int  scaleColByDivisor(ch, scaleF)
ColHeader_t *ch;
double       scaleF;
#else
void scaleColByDivisor(ColHeader_t *ch, double scaleF)
#endif
{
  ElementNode_t  *e;

  e = ch->celems;
  while (e != NULL) {  
    e->value = e->value / scaleF;
    e = e->nrowE;
  }/*while*/
}

/* ######################################################################## */
#ifdef OLDC
ElementNode_t *newElement(row, col, value)
Index_t   row;
Index_t   col;
double    value;
#else
ElementNode_t *newElement(Index_t row, Index_t col, double value)
#endif
{
  ElementNode_t  *newE;

  newE = (ElementNode_t *) malloc( sizeof(ElementNode_t) );
  if (newE == NULL) {
    error("newElement: out of memory");
    return NULL;
  }/*then*/

  /* init its fields... */
  newE->ncolE = NULL;
  newE->pcolE = NULL;
  newE->nrowE = NULL;
  newE->prowE = NULL;
  newE->row   = row;
  newE->col   = col;
  newE->value = value;

  return newE;
}

