/* sparse.h */

#ifndef SPARSE_H
#define SPARSE_H

#include "main.h"

/* 
 * Constants...
 */

#define PivotThreshold         0.001 /* value used by SPICE */
#define RescaleCount   50  /* rescale matrix after factoring this # of cols */ 


/* 
 * Data types...
 *
 * A sparse matrix has a disjoint list of col header nodes and a list
 *   of row header nodes.  Each header node points to a list of row or
 *   col element nodes; these lists are not disjoint, since elements 
 *   will often be common to both a row and a column (after all, it's
 *   a matrix :-).
 *
 * NOTE: it is assumed by the factor, solve, etc. operations that in a
 *   given sparse matrix, not only are the matrix elements stored, but 
 *   also the RHS elements b.  These form the last column of the matrix
 *   (thus, all the rows should have an entry in the last column).
 */

typedef  short  Index_t;  /* for indexing into a sparse array */

/* header node for entire sparse matrix */
typedef struct SparseMatrix {

    struct RowHeader  *rows;
    struct ColHeader  *cols;
  } SparseMatrix_t;

/* head of each row begins with one of these header nodes */
typedef struct RowHeader {

    Index_t              row;
    struct ElementNode  *relems;
    struct RowHeader    *nrowH;
    struct RowHeader    *prowH;
  } RowHeader_t;

/* head of each col begins with one of these header nodes */
typedef struct ColHeader {

    Index_t   col;
    Index_t   whichUnknown;    /* which unknown does this col represent */
      /* NOTE: we need "whichUnknown" because we may swap columns.      */
    double colScaleFactor;     /* when scaling is done by column...     */
    struct ElementNode  *celems;
    struct ColHeader    *ncolH;
    struct ColHeader    *pcolH;
  } ColHeader_t;

/* each element in the matrix */
typedef struct ElementNode {

    Index_t   row, col;
    long      markowitzProduct;
    double    value;
    struct ElementNode *nrowE;
    struct ElementNode *prowE;
    struct ElementNode *ncolE;
    struct ElementNode *pcolE;
 } ElementNode_t;


/* ######################################################################## */
#ifdef ASSERTIONS

First, the axioms for the non-previous fields:  (this is taken from 
the ICCL paper, see "examples.tex")

    /* element nodes */
    forall nodes p, p.nrowE <> p.ncolE.
    forall distinct nodes p and q, p.nrowE <> q.nrowE.
    forall distinct nodes p and q, p.ncolE <> q.ncolE.
    forall nodes p, p.(ncolE)* <> p.(nrowE)+(ncolE)*.
    forall nodes p, p.(nrowE)* <> p.(ncolE)+(nrowE)*.

    /* row and col header nodes */
    forall distinct nodes p and q, p.nrowH <> q.nrowH.
    forall distinct nodes p and q, p.ncolH <> q.ncolH.
    forall distinct nodes p and q, p.relems(ncolE)* <> q.relems(ncolE)*.
    forall distinct nodes p and q, p.celems(nrowE)* <> q.celems(nrowE)*.

    /* main sparse matrix node */
    forall distinct nodes p and q, p.rows <> q.nrowH.
    forall distinct nodes p and q, p.cols <> q.ncolH.

    /* acyclicness */
    forall nodes p, p.(rows|cols|relems|celems|nrowH|ncolH|nrowE|ncolE)+ <> p.

Second, we consider the previous fields, which cause cycles:

    /* element nodes */
    forall nodes p, p.prowE <> p.pcolE.
    forall nodes p, p.nrowE <> p.(prowE|pcolE).
    forall nodes p, p.ncolE <> p.(prowE|pcolE).
    forall distinct nodes p and q, p.prowE <> q.prowE.
    forall distinct nodes p and q, p.pcolE <> q.pcolE.

    /* disjointness of lists from an element node */
    forall nodes p, p.(ncolE)* <> p.(prowE)+(ncolE)*.
    forall nodes p, p.(nrowE)* <> p.(pcolE)+(nrowE)*.
    forall nodes p, p.(pcolE)* <> p.(nrowE)+(pcolE)*.
    forall nodes p, p.(prowE)* <> p.(ncolE)+(prowE)*.
    forall nodes p, p.(pcolE)* <> p.(prowE)+(pcolE)*.
    forall nodes p, p.(prowE)* <> p.(pcolE)+(prowE)*.

    /* cyclic relationsihps for element nodes */
    forall nodes p, p.(nrowE)(prowE) = p.
    forall nodes p, p.(prowE)(nrowE) = p.
    forall nodes p, p.(ncolE)(pcolE) = p.
    forall nodes p, p.(pcolE)(ncolE) = p.

    /* row and col header nodes */
    forall nodes p, p.nrowH <> p.prowH.
    forall distinct nodes p and q, p.prowH <> q.prowH.
    forall nodes p, p.ncolH <> p.pcolH.
    forall distinct nodes p and q, p.pcolH <> q.pcolH.

    /* cyclic relationships for row and col header nodes */
    forall nodes p, p.(nrowH)(prowH) = p.
    forall nodes p, p.(prowH)(nrowH) = p.
    forall nodes p, p.(ncolH)(pcolH) = p.
    forall nodes p, p.(pcolH)(ncolH) = p.

    /* acyclicness */
    forall nodes p, p.(prowE|pcolE|prowH|pcolH)+ <> p.
    /* NOTE: other acyclicness that we aren't addressing directly: e.g. */
    /*   going across a row, back up, and over, will not cycle around.  */
    /*   However, handled by disjointness of lists?  But are there cases*/
    /*   where we need direct axioms? */

#endif
/* ######################################################################## */


/* 
 * Prototypes...
 */

#ifdef OLDC

SparseMatrix_t *sparseCreate();
SparseMatrix_t *sparseDup();
long            sparseNumElements();
Index_t         sparseNumRows();
Index_t         sparseMaxRow();
Index_t         sparseNumCols();
Index_t         sparseMaxCol();

boolean sparseGet();
boolean sparsePut();
boolean sparseScale();
boolean doScaling();
boolean sparseFactor();
boolean sparseSolve();

/* LOCAL routines */
RowHeader_t   *getLastRowHPtr();
ColHeader_t   *getLastColHPtr();
RowHeader_t   *getThisRowHeader();
ColHeader_t   *getThisColHeader();
int            computeMarkowitzProducts();
ElementNode_t *findBestPivotElement();
                        
int     unlinkFromRow();
int     unlinkFromColumn();
int     insertIntoColumn();
int     linkIntoColumnBetween();
int     insertIntoRow();
int     linkIntoRowBetween();
ElementNode_t *swapRowsAndCols();
int     scaleRowsDueToPivot();
boolean addFillIns();
int     eliminateColUpdateRows();
double  findMaxInRow();
double  findMaxInCol();
int     scaleRowByDivisor();
int     scaleColByDivisor();
ElementNode_t  *newElement();

#else  /* define ANSI C function prototypes */

SparseMatrix_t *sparseCreate(void);
SparseMatrix_t *sparseDup(SparseMatrix_t *sm);
void            sparseFree(SparseMatrix_t *sm);
long            sparseNumElements(SparseMatrix_t *sm);
Index_t         sparseNumRows(SparseMatrix_t *sm);
Index_t         sparseMaxRow(SparseMatrix_t *sm);
Index_t         sparseNumCols(SparseMatrix_t *sm);
Index_t         sparseMaxCol(SparseMatrix_t *sm);

boolean sparseGet(SparseMatrix_t *sm, Index_t row, Index_t col, double *valuep);
boolean sparsePut(SparseMatrix_t *sm, Index_t row, Index_t col, double value);
boolean sparseScale(SparseMatrix_t *sm);
boolean doScaling(SparseMatrix_t *sm, boolean rescale);
boolean sparseFactor(SparseMatrix_t *sm);
boolean sparseSolve(SparseMatrix_t *sm, double x[]);
void    sparsePrint(SparseMatrix_t *sm);
#ifdef DEBUG 
void    sparseCheckInvariants(SparseMatrix_t *sm, long maxRow, long maxCol);
#endif

/* LOCAL routines */
RowHeader_t   *getLastRowHPtr(SparseMatrix_t *sm); 
ColHeader_t   *getLastColHPtr(SparseMatrix_t *sm);
RowHeader_t   *getThisRowHeader(SparseMatrix_t *sm, Index_t row);
ColHeader_t   *getThisColHeader(SparseMatrix_t *sm, Index_t col);
void           computeMarkowitzProducts(SparseMatrix_t *sm, 
                        RowHeader_t *pivotRH, ColHeader_t *pivotCH);
ElementNode_t *findBestPivotElement(SparseMatrix_t *sm, 
                        RowHeader_t *pivotRH, ColHeader_t *pivotCH);
                        
void    unlinkFromRow(RowHeader_t *rh, ElementNode_t *e);
void    unlinkFromColumn(ColHeader_t *ch, ElementNode_t *e);
void    insertIntoColumn(ColHeader_t *ch, ElementNode_t *e);
void    linkIntoColumnBetween(ColHeader_t *ch, ElementNode_t *prevE,
                 ElementNode_t *curE, ElementNode_t *newE);
void    insertIntoRow(RowHeader_t *rh, ElementNode_t *e);
void    linkIntoRowBetween(RowHeader_t *rh, ElementNode_t *prevE,
                 ElementNode_t *curE, ElementNode_t *newE);
ElementNode_t *swapRowsAndCols(SparseMatrix_t *sm, RowHeader_t *pivotRH,
                 ColHeader_t *pivotCH, ElementNode_t *realPivotE);
void    scaleRowsDueToPivot(ElementNode_t *pivotE);
boolean addFillIns(SparseMatrix_t *sm, ElementNode_t *pivotE);
void    eliminateColUpdateRows(ElementNode_t *pivotE);
double  findMaxInRow(RowHeader_t *rh);
double  findMaxInCol(ColHeader_t *ch);
void    scaleRowByDivisor(RowHeader_t *rh, double scaleF);
void    scaleColByDivisor(ColHeader_t *ch, double scaleF);
ElementNode_t  *newElement(Index_t row, Index_t col, double value);

#endif

#endif

