/******************************************************************* 
 *  Health.c : Model of the Columbian Health Care System           *
 *******************************************************************/ 
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "health.h"
#include <assert.h>

     
struct Village *alloc_tree(int level, int label, struct Village *back) 
{
  int                    i;
  struct Village         *new;
  
  if (level == 0) 
    return NULL;
 
  new = (struct Village *)malloc(sizeof(struct Village));
  for (i = 0; i < 4; i++) 
    new->forward[i] = alloc_tree(level - 1, (label * 4) + i + 1, new); 
  
  new->back = back;
  new->label = label;
  new->seed = label * (IQ + seed); 
  new->hosp.personnel = new->hosp.free_personnel = (int)pow(2.0, (double)level - 1);
  new->hosp.num_waiting_patients = 0;
  
  return new; 
}
  

float get_total_hosps(struct Village *village)
{
  int                    i;
  struct List            *list;
  float                  number = 0;

  if (village == NULL) return 0;
  for (i = 0; i < 4; i++)
    number += get_total_hosps(village->forward[i]);
  list = village->returned.forward;
  while (list != NULL) {
    number += (float)(list->patient->hosps_visited);
    list = list->forward; }

  return number;
}


float get_total_time(struct Village *village)
{
  int                    i;
  struct List            *list;
  float                  number = 0.0;

  if (village == NULL) return 0;
  for (i = 0; i < 4; i++)
    number += get_total_time(village->forward[i]);
  list = village->returned.forward;
  while (list != NULL) {
    number += (float)(list->patient->time);
    list = list->forward; }

  return number;
}


float get_num_people(struct Village *village) 
{
  int                    i;
  struct List            *list;
  float                  number = 0.0;
  
  if (village == NULL) return 0;
  for (i = 0; i < 4; i++)
    number += get_num_people(village->forward[i]);
  list = village->returned.forward;
  while (list != NULL) {
    number += 1.0;
    list = list->forward; }
  
  return number; 
}


void check_patients_inside(struct Village *village, struct List *list) 
{
  while (list != NULL) {
    list->patient->time_left--;
    if (list->patient->time_left == 0) {
      village->hosp.free_personnel++;
      removeList(&(village->hosp.inside), list->patient); 
      addList(&(village->returned), list->patient); 
      
      /* printf("Person being returned from hospital %d to village %d.\n",
	     village->label, list->patient->home_village->label); 
      printf("He was in for %d units of time.\n", list->patient->time); */ }
    
    list = list->forward; 
  } 
}


void check_patients_assess(struct Village *village, struct List *list) 
{
  float                 rand;
  
  while (list != NULL) {
    list->patient->time_left--;
    if (list->patient->time_left == 0) { 
      rand = my_rand(village->seed);
      village->seed = (long)(rand * IM);
      if (rand > 0.1 || village->label == 0) {
	removeList(&(village->hosp.assess), list->patient);
	addList(&(village->hosp.inside), list->patient);
	list->patient->time_left = 10;
	list->patient->time += 10; }
      else {
	village->hosp.free_personnel++;
	removeList(&(village->hosp.assess), list->patient);
	put_in_hosp(&(village->back->hosp), list->patient); } }
    
    list = list->forward; } 
}
   
  
void check_patients_waiting(struct Village *village, struct List *list) 
{
  while (list != NULL) {
    if (village->hosp.free_personnel > 0) {
      village->hosp.free_personnel--;
      list->patient->time_left = 3;
      list->patient->time += 3;
      removeList(&(village->hosp.waiting), list->patient);
      addList(&(village->hosp.assess), list->patient); }
    else 
      list->patient->time++;
    list = list->forward; } 
}


void put_in_hosp(struct Hosp *hosp, struct Patient *patient) 
{  
  patient->hosps_visited++;
   
  if (hosp->free_personnel > 0) {
    hosp->free_personnel--;
    addList(&(hosp->assess), patient); 
    patient->time_left = 3;
    patient->time += 3; }
  
  else addList(&(hosp->waiting), patient); 
}

  
struct Patient *generate_patient(struct Village *village) 
{
  struct Patient *patient;
  float rand;
  
  rand = my_rand(village->seed);
  village->seed = (long)(rand * IM);
  if (rand > 0.666) {
    patient = (struct Patient *)malloc(sizeof(struct Patient));
    patient->hosps_visited = 0;
    patient->time = 0;
    patient->time_left = 0;
    patient->home_village = village; 
    return patient; }
  return NULL; 
}
    

main(int argc, char *argv[]) 
{ 
  struct Village         *top;
  long                   i;
  
  dealwithargs(argc, argv); 
  top = NULL;
  top = alloc_tree(max_level, 0, top);
  
  printf("\n\n    Columbian Health Care Simulator\n\n");
  printf("Working...\n");
  
  for (i = 0; i < max_time; i++) {
#ifdef OUTPUT
    if (i % 50 == 0)
      printf("%d\n", i);
#endif
    sim(top);   }
  printf("Done.");
  printf("\n\n\n# of people treated:              %f people\n",
	 get_num_people(top));
  printf("Average length of stay:           %f time units\n", 
	 get_total_time(top) / get_num_people(top));
  printf("Average # of hospitals visited:   %f hospitals\n\n\n",
	 get_total_hosps(top) / get_num_people(top));
}


void sim(struct Village *village)
{
  int                    i;
  struct Patient         *patient;
  
  if (village == NULL) return;
  
  for (i = 0; i < 4; i++)
    sim(village->forward[i]);

  check_patients_inside(village, village->hosp.inside.forward);
  check_patients_assess(village, village->hosp.assess.forward);
  check_patients_waiting(village, village->hosp.waiting.forward);

/*** Generate new patients ***/  
  if ((patient = generate_patient(village)) != NULL) {  
    put_in_hosp(&(village->hosp), patient);
  }
}






























