/* getparam.c */

/*
 * Get parameter module of hierarchical N-body code; gets parameters
 *   from user for upcoming simulation run.  
 *
 * Public routines:  initparam()
 *                   getparam()
 *                   getiparam()
 *                   getrparam()
 *                   getdparam()
 *
 * Modified By:  Joe Hummel, UC-Irvine, 1992
 *
 * Original Author:
 * Copyright (c) 1991, Joshua E. Barnes, Honolulu, HI.
 * 	    It's free because it's yours.
 */

#include <string.h>

#include "defs.h"
#include "util.h"

#include "getparam.h"


/*
 * Local globals...
 */

local string *g_defaults = NULL;	/* vector of "name=value" strings */


/* ######################################################################### */
/*
 * INITPARAM: ignore arg vector, remember defaults.
 */

void initparam(string *argv, string *defv)  /* argument and default vectors */
{
    g_defaults = defv;
}


/* ######################################################################### */
/*
 * GETPARAM: prompts user for value, returns value as a string.  If no
 *   value is given and no default exists, NULL is returned.
 */

string getparam(string name)  /* name of parameter */
{
    int     i, len;
    string  def;
    char    buf[256];

    if (g_defaults == NULL)
	  error("getparam: called before initparam\n");
	  
    i = scanbind(g_defaults, name);     /* find name in defaults */
    if (i < 0)
	  error("getparam: %s unknown\n"/* Rakesh, name */);
	  
    def = extrvalue(g_defaults[i]);     /* extract default value */
    if (def != NULL)
	  fprintf(stderr, "enter %s [%s]: ", name, def);
    else
	  fprintf(stderr, "enter %s: ", name);
	  
    gets(buf);                          /* read users response   */
    len = strlen(buf) + 1;
    if (len > 1)                        /* return user's value or*/
	  return strcpy((string) malloc(len), buf);
	  
	return def;  /* otherwise return ptr to default value (or NULL) */
}


/* ######################################################################### */
/*
 * GETIPARAM, ..., GETRPARAM: get int, bool, or real parameters.
 */

int32 getiparam(string name)  /* name of parameter */
{
    string  val;

    for (val = NULL; val == NULL; )  /* while nothing input    */
	  val = getparam(name);         /* obtain value from user */
	
	if (sizeof(int) == 4)           /* convert to an integer  */
      return atoi(val);
      
    return atol(val);
}


bool getbparam(string name)  /* name of parameter */
{
    string  val;

    for (val = NULL; val == NULL; )  /* get value from user */
	  val = getparam(name);
	  
    if (strchr("tTyY1", *val) != NULL)  /* is value true? */
        return TRUE;
    if (strchr("fFnN0", *val) != NULL)  /* is value false? */
        return FALSE;
        
    error("getbparam: %s=%s not bool\n"/* Rakesh , name, val */);
    return FALSE;  /* useless, but don't get a compiler warning */
}


real getrparam(string name)  /* name of parameter */
{
    string  val;

    for (val = NULL; val == NULL; )  /* get value from user */
	  val = getparam(name);
	
    return atof(val);  /* convert to double, then C will cast to return type */
}


/**************************** LOCAL ROUTINES *********************************/

/* ######################################################################### */
/*
 * SCANBIND: scan binding vector for name, return index (-1 if not found).
 */

local int scanbind(string bvec[], string name)
{
    int i;

    for (i = 0; bvec[i] != NULL; i++)  /* search loop */
	  if (matchname(bvec[i], name))
        return (i);
	    
    return -1;  /* otherwise not found */
}


/* ######################################################################### */
/*
 * MATCHNAME: determine if "name=value" matches "name".  Returns C's def
 *   of T or F.
 */

local bool matchname(string bind, string name)
{
    char *bp, *np;

    bp = bind;
    np = name;
    while (*bp == *np) {  /* keep going as long as there's a match */
	  bp++;
	  np++;
    }/*while*/
    
    return (*bp == '=') And (*np == EOS);  /* must be at "=" and end of name */
}


/* ######################################################################### */
/*
 * EXTRVALUE: extract value from name=value string; i.e. return ptr to thing
 *   after the "=", otherwise NULL.
 */

local string extrvalue(string arg)  /* string of the form "name=value" */
{
    while (*arg != EOS)     /* until we reach end of string... */
	  if (*arg++ == '=')    /* found = yet?                    */
        break;              /* if so, exit and see if anything follows... */
    
    if (*arg != EOS)  /* does anything follow after the = sign */
      return arg;   
      
    return NULL;  /* otherwise not found... */
}
