/* grav.c */

/*
 * Gravity module for hierarchical N-body code; routines to compute
 *   gravity.
 *
 * Public routines:  hackgrav()
 *
 * Modified By:  Joe Hummel, UC-Irvine, 1992
 *
 * Original Author:
 * Copyright (c) 1991, Joshua E. Barnes, Honolulu, HI.
 * 	    It's free because it's yours.
 */

#include "defs.h"
#include "globals.h"

#include "grav.h"


/* ######################################################################### */
/*
 * HACKGRAV: evaluate grav field at a given particle.
 */

local NodePtr_t  g_pskip;  /* body to skip in force evaluation */
local vector     g_pos0;   /* point at which to evaluate field */
local real       g_phi0;   /* computed potential at pos0       */
local vector     g_acc0;   /* computed acceleration at pos0    */

void hackgrav(NodePtr_t p)
{
    g_pskip = p;                 /* exclude from force calc. */
    SETV(g_pos0, Pos(p));        /* eval force on this point */
    g_phi0 = 0.0;                /* init grav. potential and */
    CLRV(g_acc0);                /*   force accumulators     */

    g_n2bterm = g_nbcterm = 0;   /* zero the term counters   */
    g_skipself = FALSE;          /* reset flag to look for p */

    hackwalk();                  /* recursively scan tree    */
    Phi(p) = g_phi0;             /* stash resulting pot. and */
    SETV(Acc(p), g_acc0);        /*   acceleration in body p */
}


/**************************** LOCAL ROUTINES *********************************/

/* ######################################################################### */
/*
 * GRAVSUB: compute a single body-body or body-cell interaction.
 */

local NodePtr_t  g_pmem;    /* for memorized data to be shared */
local vector     g_dr;      /* between gravsub and subdivp     */
local real       g_drsq;

local void gravsub(NodePtr_t p)  /* body or cell to interact with */
{
    vector  ai, quaddr;
    real    drabs, phii, mor3, dr5inv, phiquad, drquaddr;

    if (p != g_pmem) {              /* cant use memorized data? */
      SUBV(g_dr, Pos(p), g_pos0);   /* then find seperation     */
      DOTVP(g_drsq, g_dr, g_dr);    /* and square of distance   */
    }/*then*/
    
    g_drsq += g_eps*g_eps;          /* use standard softening   */
    drabs = sqrt((double) g_drsq);  /* find norm of distance    */
    phii = Mass(p) / drabs;         /* and contribution to phi  */
    g_phi0 -= phii;                 /* add to total potential   */
    mor3 = phii / g_drsq;           /* form mass / radius qubed */
    MULVS(ai, g_dr, mor3);          /* and contribution to acc. */
    ADDV(g_acc0, g_acc0, ai);       /* add to net acceleration  */

    if(Type(p) == CELL)             /* a body-cell interaction? */
      g_nbcterm++;                  /* count body-cell term     */
    else                            /* a body-body interaction */
	  g_n2bterm++;                  /* count body-body term    */
}


/* ######################################################################### */
/*
 * HACKWALK: walk the tree, opening cells too close to a given point.
 */

local real  g_tolsq;

local void hackwalk()
{
    g_tolsq = g_tol * g_tol;
    walksub(g_root, g_rsize * g_rsize);
}


/* ######################################################################### */
/*
 * WALKSUB: recursive routine to do hackwalk operation.
 *
 * nodeptr p   -- pointer into body-tree
 * real    dsq -- size of box squared 
 */

local void walksub(NodePtr_t p, real dsq)
{
    register int32  k;
    
#ifdef DEBUG
      printf("walksub: p = %o  dsq = %f\n", p, dsq);
#endif

    if (subdivp(p, dsq)) {                   /* should p be opened?      */
    
        for (k = 0; k < NSUB; k++)           /* loop over the subcells   */
          if (Subp(p, k) != NULL)            /* does this one exist?     */
            walksub(Subp(p, k), dsq / 4.0);  /* examine it in turn       */
    }/*then*/
    else if (p != g_pskip)                   /* should p be included?    */
           gravsub(p);                       /*   then use interaction   */
         else                                /* self-interact'n skipped? */
	       g_skipself = TRUE;                /*   note calculation is OK */
}


/* ######################################################################### */
/*
 * SUBDIVP: decide if a node should be opened.
 *
 * Side effects: sets pmem, dr, and drsq.
 *
 * nodeptr p   -- body/cell to be tested
 * real    dsq -- size of cell squared 
 */

local bool subdivp(NodePtr_t p, real dsq)
{
    if (Type(p) == BODY)               /* at tip of tree?       */
      return (FALSE);                  /*   then cant subdivide */
      
    SUBV(g_dr, Pos(p), g_pos0);        /* compute displacement  */
    DOTVP(g_drsq, g_dr, g_dr);         /* and find dist squared */
    g_pmem = p;                        /* remember we know them */
    return (g_tolsq * g_drsq) < dsq;   /* use geometrical rule  */
}

