/* io.c */

/*
 * I/O module for hierarchical N-body code.
 *
 * Public routines:  inputdata()
 *                   initoutput()
 *                   stopoutput()
 *                   output()
 *
 * Modified By:  Joe Hummel, UC-Irvine, 1992
 *
 * Original Author:
 * Copyright (c) 1991, Joshua E. Barnes, Honolulu, HI.
 * 	    It's free because it's yours.
 */

#include "defs.h"
#include "globals.h"
#include "util.h"

#include "io.h"


/* ######################################################################## */
/*
 * INPUTDATA: read initial conditions from input file.  Should only be
 *   called if input file exists.
 */

void inputdata()
{
    permanent char headbuf[256];
    
    stream    instr;
    int32     i, ndim;
    register  NodePtr_t  p;

    instr = fopen(g_infile, "r");  /* open input stream */
    if (instr == NULL)
	  error("inputdata: cannot find file %s\n"/* Rakesh , g_infile */);
	  
    sprintf(headbuf, "Hack code: input file %s\n", g_infile);
    g_headline = headbuf;
    in_int(instr, &g_nbody);
    if (g_nbody < 1)
	  error("inputdata: nbody is absurd\n");
	  
    in_int(instr, &ndim);
    if (ndim != NDIM)
	  error("inputdata: ndim is absurd\n");
	  
    in_real(instr, &g_tnow);

	/*
	 * Loop over new bodies...
	 */
    g_particles = NULL;
    for (i = 0; i < g_nbody; i++) {/* allocate nodes for bodies  */
    
      p = (NodePtr_t) malloc(sizeof(Node_t)); /* allocate a node for 1 body */
      if (p == NULL)                          /* check space is available   */
        error("inputdata: not enuf memory\n");
        
	  Next(p) = g_particles;       /* link in at head of list... */
	  g_particles = p;

	  Type(p) = BODY;
	}/*for*/
	
	p = g_particles;
	while (p != NULL) {            /* read in masses... */
	  in_real(instr, &Mass(p));
	  p = Next(p);
	}/*while*/
	
	p = g_particles;
	while (p != NULL) {            /* read in positions... */
	  in_vector(instr, Pos(p));
	  p = Next(p);
	}/*while*/
	
	p = g_particles;
	while (p != NULL) {            /* read in velocities... */
	  in_vector(instr, Vel(p));
	  p = Next(p);
	}/*while*/

    fclose(instr);  /* close input stream */
}


/* ######################################################################## */
/*
 * INITOUTPUT: initialize output routines.
 */

local stream  g_outstr;  /* output stream pointer */

void initoutput()
{
    printf("\n\t\t%s\n\n", g_headline);   /* print headline, params */
    printf("%12s%12s%12s%12s%12s%12s\n",
           "nbody", "dtime", "eps", "tol", "dtout", "tstop");
    
    OUT_INT32(stdout, g_nbody, "12");
    printf("%12.5f%12.4f%12.2f%12.3f%12.2f\n\n",
           g_dtime, g_eps, g_tol, g_dtout, g_tstop);
           
    if (g_outfile != NULL) {               /* output file specified? */
        g_outstr = fopen(g_outfile, "w");  /* setup output stream    */
	    if (g_outstr == NULL)
	      error("initoutput: cannot open file %s\n"/* Rakesh , g_outfile*/);
    } 
    else g_outstr = NULL;                  /* turn off data output   */
}


/* ######################################################################## */
/*
 * STOPOUTPUT: finish up after a run.
 */

void stopoutput()
{
    if (g_outstr != NULL)
      fclose(g_outstr);
}


/* ######################################################################## */
/*
 * OUTPUT: compute diagnostics and output data.
 */

#ifndef TIME_ONLY

void output()
{
    register NodePtr_t  p;
    
    int32    nttot, nbavg, ncavg;
    real     etot[3];               /* binding, kinetic, potential energy */
    vector   cmphase[2];            /* center of mass coordinates */
    vector   amvec;                 /* angular momentum vector */

    diagnostics(etot, cmphase, amvec);    /* compute std diagnostics  */
    
    nttot = g_n2bcalc + g_nbccalc;
    nbavg = (int32) ((real) g_n2bcalc / (real) g_nbody);
    ncavg = (int32) ((real) g_nbccalc / (real) g_nbody);
    
    printf("\n%9s%9s%9s%9s%9s%9s%9s%9s\n", "tnow", "T+U",
	   "T/U", "nttot", "nbavg", "ncavg", "selfint", "cputime");
    printf("%9.3f%9.4f%9.4f", g_tnow, etot[0], etot[1]/etot[2]);
	OUT_INT32(stdout, nttot, "9");
	OUT_INT32(stdout, nbavg, "9");
	OUT_INT32(stdout, ncavg, "9");
	OUT_INT32(stdout, g_selfint, "9");
#ifdef CPUTIME
    printf("%9.2f\n\n", cputime());
#else
    printf("%9.2f\n\n", 0.00);
#endif
	
    printvec("cm pos", cmphase[0]);
    printvec("cm vel", cmphase[1]);
    printvec("am vec", amvec);
    
    if ((g_outstr != NULL) And                   /* output file given and */
	    ((g_tout - 0.01 * g_dtime) <= g_tnow)) { /* time for next output? */
	    
	  out_int(g_outstr, g_nbody);                /* write particle data...*/
	  out_int(g_outstr, NDIM);
	  out_real(g_outstr, g_tnow);
	  
	  p = g_particles;
	  while (p != NULL) { 
	    out_real(g_outstr, Mass(p));
	    p = Next(p);
	  }/*while*/
	  
	  p = g_particles;
	  while (p != NULL) { 
	    out_vector(g_outstr, Pos(p));
	    p = Next(p);
	  }/*while*/

	  p = g_particles;
	  while (p != NULL) { 
	    out_vector(g_outstr, Vel(p));
	    p = Next(p);
	  }/*while*/
	  
	  printf("\n\tparticle data written\n");    /* report data written    */
	  g_tout += g_dtout;                        /* schedule next data out */
    }/*then*/
}

#endif


/**************************** LOCAL ROUTINES *********************************/

/* ######################################################################## */
/*
 * DIAGNOSTICS: compute set of dynamical diagnostics.
 */

#ifndef TIME_ONLY

local void diagnostics(real etot[3], vector cmphase[2], vector amvec)
{
    register NodePtr_t p;
 
    real    mtot;    /* total mass of N-body system */
    matrix  keten;   /* kinetic energy tensor */
    matrix  peten;   /* potential energy tensor */

    real    velsq;
    vector  tmpv;
    matrix  tmpt;

    mtot = 0.0;               /* zero total mass        */
    etot[1] = etot[2] = 0.0;  /* zero total KE and PE   */
    CLRM(keten);              /* zero ke tensor         */
    CLRM(peten);              /* zero pe tensor         */
    CLRV(cmphase[0]);         /* zero c. of m. position */
    CLRV(cmphase[1]);         /* zero c. of m. velocity */
    CLRV(amvec);              /* zero am vector         */

    /*
     * Loop over all the particles...
     */
    p = g_particles;
    while (p != NULL) {

	  mtot += Mass(p);                      /*   sum particle masses    */
	  DOTVP(velsq, Vel(p), Vel(p));         /*   square vel vector      */
	  etot[1] += 0.5 * Mass(p) * velsq;     /*   sum current KE         */
	  etot[2] += 0.5 * Mass(p) * Phi(p);    /*   and current PE         */
	  MULVS(tmpv, Vel(p), 0.5 * Mass(p));   /*   sum 0.5 m v_i v_j      */
	  OUTVP(tmpt, tmpv, Vel(p));
	  ADDM(keten, keten, tmpt);
	  MULVS(tmpv, Pos(p), Mass(p));         /*   sum m r_i a_j          */
	  OUTVP(tmpt, tmpv, Acc(p));
	  ADDM(peten, peten, tmpt);
	  MULVS(tmpv, Pos(p), Mass(p));         /*   sum cm position        */
	  ADDV(cmphase[0], cmphase[0], tmpv);
	  MULVS(tmpv, Vel(p), Mass(p));         /*   sum cm momentum        */
	  ADDV(cmphase[1], cmphase[1], tmpv);
	  CROSSVP(tmpv, Pos(p), Vel(p));        /*   sum angular momentum   */
	  MULVS(tmpv, tmpv, Mass(p));
	  ADDV(amvec, amvec, tmpv);
	  
	  p = Next(p);
    }/*while*/
    
    etot[0] = etot[1] + etot[2];                /* sum KE and PE        */
    DIVVS(cmphase[0], cmphase[0], mtot);    /* normalize cm coords  */
    DIVVS(cmphase[1], cmphase[1], mtot);
}

#endif


/* ######################################################################## */
/*
 * PRINTVEC:
 */
 
local void printvec(string name, vector vec)
{
    printf("\t %9s%9.4f%9.4f%9.4f\n", name, vec[0], vec[1], vec[2]);
}


/* ######################################################################## */
/*
 * Low-level input and output operations.
 */

local void in_int(stream str, int32 *iptr)
{
    if (IN_INT32(str,iptr) != 1)
	  error("in_int: input conversion error\n");
}

local void in_real(stream str, real *rptr)
{
    double tmp;  /* always input as double, convert to real if necessary */

    if (fscanf(str, "%lf", &tmp) != 1)
	  error("in_real: input conversion error\n");
	
    *rptr = (real) tmp;  /* convert to desired return type */
}

local void in_vector(stream str, vector vec)
{
    double tmpx, tmpy, tmpz;

    if (fscanf(str, "%lf%lf%lf", &tmpx, &tmpy, &tmpz) != 3)
	  error("in_vector: input conversion error\n");
	  
    vec[0] = tmpx;
    vec[1] = tmpy;    
    vec[2] = tmpz;
}

#ifndef TIME_ONLY

local void out_int(stream str, int32 ival)
{
    fprintf(str, "  ");
    OUT_INT32(str, ival, "");  /* no field size ==> default size */
    fprintf(str, "\n");
}

local void out_real(stream str, real rval)
{
    fprintf(str, " %21.14E\n", (double) rval);
}

local void out_vector(stream str, vector vec)
{
    fprintf(str, " %21.14E %21.14E %21.14E\n", vec[0], vec[1], vec[2]);
}

#endif

