/* load.c */

/*
 * Load module for hierarchical N-body code; routines to create body-tree.
 *
 * Public routines:  maketree()
 *
 * Modified By:  Joe Hummel, UC-Irvine, 1992
 *
 * Original Author:
 * Copyright (c) 1991, Joshua E. Barnes, Honolulu, HI.
 * 	    It's free because it's yours.
 */

#include "defs.h"
#include "globals.h"
#include "util.h"

#include "load.h"


/*
 * local globals...
 */
local NodePtr_t  g_cells = NULL;      /* all cells kept on this list...    */
local NodePtr_t  g_free  = NULL;      /* free cells partial sublist        */


/* ######################################################################### */
/*
 * MAKETREE: initialize tree structure for hack force calculation.
 *
 * bodyptr bodies  -- ptr to particles list to build into tree 
 * int     nbody -- number of bodies in above array
 */

void maketree(NodePtr_t bodies, int32 nbody)
{
    register NodePtr_t  p;

    g_root = NULL;           /* deallocate current tree */
    g_free = g_cells;        /* make all the cells free */
    
    p = bodies;
    while (p != NULL) { /* loop over all bodies in list */

	  if (Mass(p) != 0.0) {  /* only load massive ones  */
        expandbox(p);        /* expand root to fit      */
        loadtree(p);         /* insert into tree        */
	  }/*then*/
	  
	  p = Next(p);
	}/*while*/

    hackcofm(g_root);				     /* find c-of-m coordinates */
}


/**************************** LOCAL ROUTINES *********************************/

/* ######################################################################### */
/*
 * EXPANDBOX: enlarge cubical "box", salvaging existing tree structure.
 */

local void expandbox(NodePtr_t p)  /* body to be loaded */
{
    int32      k, xtmp[NDIM], xmid[NDIM];
    vector     rmid;
    NodePtr_t  newCell;

    while (Not intcoord(xtmp, Pos(p))) {       /* expand box (rarely)  */
    
#ifdef DEBUG
        printf("expandbox: expanding box\n");
#endif
            
      ADDVS(rmid, g_rmin, 0.5 * g_rsize);      /* find box midpoint    */
      for (k = 0; k < NDIM; k++)               /* loop over dimensions */
        if (Pos(p)[k] < rmid[k])               /* is p left of mid?    */
          g_rmin[k] -= g_rsize;                /* extend to left       */
      g_rsize = 2.0 * g_rsize;                 /* double length of box */
      
#ifdef DEBUG
        printf("\t   rmin = [%8.4f,%8.4f,%8.4f]\trsize = %8.4f\n",
                     g_rmin[0], g_rmin[1], g_rmin[3], g_rsize);
#endif
                     
      if (g_root != NULL) {                    /* repot existing tree? */
      
        newCell = makecell();                  /* create new root cell */
        assert(intcoord(xmid, rmid));          /* locate old root cell */
        k = subindex(xmid, IMAX >> 1);         /* find old tree index  */
        Subp(newCell, k) = g_root;             /* graft old on new     */
        g_root = newCell;                      /* plant new tree       */
      }/*then*/
      
    }/*while*/
}


/* ######################################################################### */
/*
 * LOADTREE: descend tree and insert particle.
 */

local void loadtree(NodePtr_t p)  /* body to load into tree */
{
    int32       l, xp[NDIM], xq[NDIM];
    NodePtr_t  *qptr, c;

    assert(intcoord(xp, Pos(p)));   /* form integer coords  */
    l = IMAX >> 1;                  /* start with top bit   */
    qptr = &g_root;                 /* start with tree root */
    while (*qptr != NULL) {         /* loop descending tree */
    
#ifdef DEBUG
        printf("loadtree: descending tree  l = %o\n", l);
#endif
	  assert(l != 0);                      /* dont run out of bits */
	  
	  if (Type(*qptr) == BODY) {           /* reached a "leaf"?    */
	  
#ifdef DEBUG
          printf("loadtree: replacing body with cell\n");
#endif
	    c = makecell();                    /* alloc a new cell     */
	    assert(intcoord(xq, Pos(*qptr)));  /* get integer coords   */
	    Subp(c, subindex(xq, l)) = *qptr;  /* put body in cell     */
	    *qptr = c;                         /* link cell in tree    */
	  }/*then*/
	
	  qptr = &Subp(*qptr, subindex(xp, l));/* move down one level  */
	  l = l >> 1;                          /* and test next bit    */
    }/*while*/
    
#ifdef DEBUG
	  printf("loadtree: installing body  l = %o\n", l);
#endif
	  
    *qptr = p;  /* found place, store p */
}


/* ######################################################################### */
/*
 * INTCOORD: compute integerized coordinates.
 *
 * Returns: TRUE unless rp was out of bounds.
 *
 * int    xp[NDIM] -- integerized coordinate vector [0,IMAX)
 * vector rp       -- real coordinate vector (system coords)
 */

local bool intcoord(int32 xp[NDIM], vector rp)
{
    register int32  k;
    bool     inb;
    double   xsc;

#ifdef DEBUG
      printf("intcoord: rp = [%8.4f,%8.4f,%8.4f]\n", rp[0], rp[1], rp[2]);
#endif
        
    inb = TRUE;                             /* use to check bounds   */
    for (k = 0; k < NDIM; k++) {            /* loop over dimensions  */
      xsc = (rp[k] - g_rmin[k]) / g_rsize;  /* scale to range [0,1)  */
      if ((0.0 <= xsc) And (xsc < 1.0))     /* within unit interval? */
        xp[k] = floor(IMAX * xsc);          /* then integerize       */
      else                                  /* out of range          */
        inb = FALSE;                        /* then remember that    */
    }/*for*/
    
#ifdef DEBUG
      printf("\t  xp = [%8x,%8x,%8x]\tinb = %d\n", xp[0], xp[1], xp[2], inb);
#endif
      
    return inb;
}


/* ######################################################################### */
/*
 * SUBINDEX: determine which subcell to select.
 *
 * int x[NDIM] -- integerized coordinates of particle
 * int l       -- current level of tree 
 */

local int subindex(int32 x[NDIM], int32 l)
{
    register int32 i, k;

#ifdef DEBUG
      printf("subindex: x = [%8x,%8x,%8x]\tl = %8x\n", x[0],x[1],x[2],l);
#endif

    i = 0;                        /* sum index in i       */
    for (k = 0; k < NDIM; k++)    /* check each dimension */
      if (x[k] & l)               /* if beyond midpoint   */
        i += NSUB >> (k + 1);     /* skip over subcells   */
        
    return i;
}


/* ######################################################################### */
/*
 * HACKCOFM: descend tree finding center-of-mass coordinates.
 */

local void hackcofm(NodePtr_t q)  /* pointer into body-tree */
{
    register  int32      i;
    register  NodePtr_t  r;
    permanent vector     tmpv, dr;
    permanent real       drsq;
    permanent matrix     drdr, Idrsq, tmpm;

    if (Type(q) == CELL) {                /* is this a cell?      */
    
      Mass(q) = 0.0;                      /* init total mass      */
      CLRV(Pos(q));                       /* and c. of m.         */
      
      for (i = 0; i < NSUB; i++) {        /* loop over subcells   */
        r = Subp(q, i);
        if (r != NULL) {                  /* does subcell exist?  */
          hackcofm(r);                    /* find subcell cm      */
          Mass(q) += Mass(r);             /* sum total mass       */
          MULVS(tmpv, Pos(r), Mass(r));   /* find moment          */
          ADDV(Pos(q), Pos(q), tmpv);     /* sum tot. moment      */
        }/*then*/
      }/*for*/
      
      DIVVS(Pos(q), Pos(q), Mass(q));     /* rescale cms position */

    }/*then*/
}


/* ######################################################################### */
/*
 * MAKECELL: allocation routine for cells.
 */

local NodePtr_t makecell()
{
    register NodePtr_t  c;
    register int32      i;

    if (g_free != NULL) {  /* then use one from the free list */
      c = g_free;
      g_free = Next(g_free);
    }/*then*/
    else {  /* no more free cells, try to allocate more */
    
      c = (NodePtr_t) malloc(sizeof(Node_t));
      if (c == NULL)
        error("maketree: not enuf memory\n");

      Next(c) = g_cells;
      g_cells = c;    
    }/*else*/
	  
    Type(c) = CELL;
    for (i = 0; i < NSUB; i++)
      Subp(c, i) = NULL;
	  
    return c;
}

