/* main.c */

/*
 * Main module of hierarchical N-body code.
 *
 * NOTE: defines that control compilation are in the file "defs.h".
 *
 * Modified By:  Joe Hummel, UC-Irvine, 1992
 *
 * Original Author:
 * Copyright (c) 1991, Joshua E. Barnes, Honolulu, HI.
 * 	    It's free because it's yours.
 */

#include "defs.h"
#include "globals.h"
#include "getparam.h"
#include "grav.h"
#include "io.h"
#include "load.h"
#include "util.h"

#include "main.h"


#ifdef COMPILE_AS_ONE    /* compile program as one big .c file? */
#include "globals.c"
#include "getparam.c"
#include "grav.c"
#include "io.c"
#include "load.c"
#include "util.c"
#endif


/* ######################################################################### */
/*
 * MAIN: 
 */
 
main(int argc, string argv[])
{
    double  start_time, stop_time;
    
    printf("** Starting Barnes-Hut Treecode (dynamic octree) **\n\n");

    initparam(argv, g_defv);  /* setup parameter access */
    startrun();               /* set params, input data */
    initoutput();             /* begin system output    */

    start_time = cputime();
    
    while (g_tnow < g_tstop + 0.1 * g_dtime)  /* while not past tstop    */
      stepsystem();                           /*   advance N-body system */
      
    stop_time = cputime();

    stopoutput();  /* finish up output */

    printf("\nTotal System and User Time: %f seconds.\n", 
           stop_time - start_time);
    printf("\n** Done **\n");

    return 0;  /* success */
}


/* ######################################################################### */
/*
 * STARTRUN: startup hierarchical N-body code.
 */

void startrun()
{
    g_infile = getparam("in");  /* set I/O file names        */
    if (g_infile != NULL)       /* was data file given?      */
	  inputdata();              /* yes, so read inital data  */
    else {  /* no, so make initial conditions...             */
    
      g_nbody = getiparam("nbody");  /* get nbody parameter  */
      if (g_nbody < 1)               /* is value absurd?     */
        error("startrun: absurd nbody\n");
	  srand(getiparam("seed"));      /* set random generator */
	  testdata();                    /* make test model      */
    }/*else*/
    
    g_outfile = getparam("out");
      /* NOTE: g_outfile could be NULL -- no file to output to */
    g_dtime   = getrparam("dtime");  /* get various parameters */
    g_eps     = getrparam("eps");
    g_tol     = getrparam("tol");
    g_tstop   = getrparam("tstop");
    g_dtout   = getrparam("dtout");

    g_nstep = 0;                 /* start counting steps  */
    g_tout  = g_tnow;            /* schedule first output */
    SETVS(g_rmin, -2.0);         /* init box scaling      */
    g_rsize = -2.0 * g_rmin[0];
}


/* ######################################################################### */
/*
 * TESTDATA: generate Plummer model initial conditions for test runs,
 * scaled to units such that M = -4E = G = 1 (Henon, Hegge, etc).
 * See Aarseth, SJ, Henon, M, & Wielen, R (1974) Astr & Ap, 37, 183.
 */

#define MFRAC  0.999  /* mass cut off at MFRAC of total */

void testdata()
{
    int32      i;
    real       rsc, vsc, r, v, x, y;
    vector     cmr, cmv;
    register   NodePtr_t  p;

    g_headline = "Hack code: Plummer model";  /* supply default headline  */
    g_tnow = 0.0;                             /* reset elapsed model time */

    rsc = 3 * PI / 16;      /* set length scale factor */
    vsc = sqrt(1.0 / rsc);  /* and recip. speed scale  */
    CLRV(cmr);              /* init cm pos, vel        */
    CLRV(cmv);

    /*
     * loop over particles...
     */
    g_particles = NULL;
    for (i = 0; i < g_nbody; i++) {
    
      p = (NodePtr_t) malloc(sizeof(Node_t)); /* allocate a node for 1 body */
      if (p == NULL)                          /* check space is available   */
        error("testdata: not enuf memory\n");
        
	  Next(p) = g_particles;      /* link in at head of list... */
	  g_particles = p;
	  
	  Type(p) = BODY;             /* tag as a body */
	  Mass(p) = 1.0 / g_nbody;    /* set masses equal */
	                              /* pick r in struct units */
	  r = 1 / sqrt(pow(xrand(0.0, MFRAC), -2.0/3.0) - 1);
	  pickshell(Pos(p), rsc * r); /* pick scaled position   */
	  ADDV(cmr, cmr, Pos(p));     /* add to running sum     */
	  
	  do {                    /* select from fn g(x)  */
	    x = xrand(0.0, 1.0);  /* for x in range 0:1   */
	    y = xrand(0.0, 0.1);  /* max of g(x) is 0.092 */
	  } while (y > x*x * pow(1 - x*x, 3.5));   /* using von Neumann tech */
	  
	  v = sqrt(2.0) * x / pow(1 + r*r, 0.25);  /* find v in struct units */
	  pickshell(Vel(p), vsc * v);		       /* pick scaled velocity   */
	  ADDV(cmv, cmv, Vel(p));			       /* add to running sum     */
	  
    }/*for*/

    DIVVS(cmr, cmr, (real) g_nbody);  /* normalize cm coords */
    DIVVS(cmv, cmv, (real) g_nbody);
    DIVVS(cmr, cmr, (real) g_nbody);  /* normalize cm coords */
    DIVVS(cmv, cmv, (real) g_nbody);

    /*
     * loop over particles again...
     */
    p = g_particles;
    while (p != NULL) {
    
	  SUBV(Pos(p), Pos(p), cmr);  /* offset by cm coords */
	  SUBV(Vel(p), Vel(p), cmv);
	  
	  p = Next(p);
    }/*while*/
}


/* ######################################################################### */
/*
 * PICKSHELL: pick a random point on a sphere of specified radius.
 *
 * real vec[] -- coordinate vector chosen
 * real rad   -- radius of chosen point
 */

void pickshell(real vec[], real rad)
{
    register int32  k;
    real     rsq, rsc;
    
    do {                            /* pick point in NDIM-space */
      for (k = 0; k < NDIM; k++)    /* loop over dimensions     */
        vec[k] = xrand(-1.0, 1.0);  /* pick from unit cube      */
	  DOTVP(rsq, vec, vec);         /* compute radius squared   */
    } while (rsq > 1.0);            /* reject if outside sphere */
    
    rsc = rad / sqrt(rsq);  /* compute scaling factor  */
    MULVS(vec, vec, rsc);   /* rescale to radius given */
}


/* ######################################################################### */
/*
 * STEPSYSTEM: advance N-body system one time-step.
 */

void stepsystem()
{
    register  NodePtr_t  p;
    real      dthf;
    vector    acc1, dacc, dvel, vel1, dpos;

    dthf = 0.5 * g_dtime;                   /* set basic half-step      */

    maketree(g_particles, g_nbody);         /* load bodies into tree    */
    g_n2bcalc = g_nbccalc = g_selfint = 0;  /* clear interaction counts */

    /*
     * loop over particles...
     */
    p = g_particles;
    while (p != NULL) {
    
	  SETV(acc1, Acc(p));            /* save old acceleration */
	  hackgrav(p);                   /* compute new acc for p */
	  if (g_nstep > 0) {             /* past the first step?  */
        SUBV(dacc, Acc(p), acc1);    /* use change in accel   */
        MULVS(dvel, dacc, dthf);     /* to make 2nd order     */
        ADDV(Vel(p), Vel(p), dvel);  /* correction to vel     */
      }/*then*/
      
      p = Next(p);
    }/*while*/

    /*
     * do major or minor output...
     */

    output();


    /*
     * advance bodies...
     */
    p = g_particles;
    while (p != NULL) {
    
	  MULVS(dvel, Acc(p), dthf);   /* use current accel'n  */
	  ADDV(vel1, Vel(p), dvel);    /* find vel at midpoint */
	  MULVS(dpos, vel1, g_dtime);  /* find pos at endpoint */
	  ADDV(Pos(p), Pos(p), dpos);  /* advance position     */
	  ADDV(Vel(p), vel1, dvel);    /* advance velocity     */
	  
	  p = Next(p);
    }/*while*/

    g_nstep++;                  /* count another step    */
    g_tnow = g_tnow + g_dtime;  /* finally, advance time */
}

