/* PUG.C  Main procedures for Parallel Unstructured Grid Control Volume Finite
   Element Method
*/

#include <string.h>

#define _PUGGLOBALS_
#include "pug.h"
#undef _PUGGLOBALS_

int itmax = ITMAX;                         /* Max iterations */
int smoothmax = SMOOTHMAX;                 /* Max smoothing iterations */
int noout = False;                         /* Produce output flag */


/* Computes relaxation of previous iteration */
void relax(nh,nfs)
	 node *nh;
	 int nfs;
{
  int i;
  double r;
  node *n;

  r = 1.0 - phis[nfs].relax;
  n = nh;

  while(n!=NULL) {

	n->ac /= phis[nfs].relax;
	n->acon += n->ac * r * n->phi[nfs];

	n=n->next;
  }
}


/* Routine to parse out options from the command line */
int parsecommandline(argc,argv)
	 int argc;
	 char *argv[];
{
  int i,a,complain,logmaxdepth = MAXDEPTH;

  complain = 0;

  for(i=1;i<argc;i++) {
	if(!strcmp(argv[i],"-savet")) savefile = argv[++i];
	else if(!strcmp(argv[i],"-loadt")) loadfile = argv[++i];
	else if(!strcmp(argv[i],"-smooth")) smoothmax = atoi(argv[++i]);
	else if(!strcmp(argv[i],"-iter")) itmax = atoi(argv[++i]);
	else if(!strcmp(argv[i],"-sweeps")) solve_sweeps = atoi(argv[++i]);
	else if(!strcmp(argv[i],"-maxdepth")) {
	  logmaxdepth = atoi(argv[++i]);
	  maxdepth = 1<<logmaxdepth;
	} else if(!strcmp(argv[i],"-mindepth")) {
	  mindepth = atoi(argv[++i]);
	} else if(!strcmp(argv[i],"-nosolve")) nosolve = 1;
	else if(!strcmp(argv[i],"-scale")) scale = 1;
	else if(!strcmp(argv[i],"-noout")) noout = 1;
	else if(!strcmp(argv[i],"-complain")) complain = 1;
	else if(!strcmp(argv[i],"-version")) {
	  printf("PUG: Version %s\n",VERSION);
	  exit(0);
	} else if(!strcmp(argv[i],"?") || !strcmp(argv[i],"-help")) {
	  printf("Usage: pug [-complain] [-savet <fn>] [-loadt <fn>]");
	  printf(" [-help] [-version] [-smooth n] [-iter n]");
	  printf(" [-maxdepth n] [-noscale] [-noout]");
	  printf(" [-sweeps n]\nPlus the usual X windows options\n");
	  printf("Options may appear in any order.\n");
	  printf(" -iter:     Number of iterations of the whole process (%d)\n",itmax);
	  printf(" -sweeps:   Number of solving iterations (%d)\n",solve_sweeps);
	  printf(" -smooth:   Number of smoothing iterations on grid (%d)\n",smoothmax);
	  printf(" -maxdepth: Max depth of the quadtree (%d)\n",logmaxdepth);
	  printf(" -mindepth: Min depth of the quadtree (%d)\n",mindepth);
	  printf(" -scale:    Calculate bounding square instead of using unit square\n");
	  printf(" -savet fn: Save triangulation as file fn\n");
	  printf(" -loadt fn: Load triangulation from file fn\n");
	  printf(" -nosolve:  Do not run the solver\n");
	  printf(" -noout:    Do not generate output\n");
	  printf(" -complain: Complain about subsequently unrecognized options (%c)\n",
			 (complain) ? 'T' : 'F');
	  printf(" -version:  Print version number and exit\n");
	  exit(0);
	} else if(complain) {
	  printf("**Unrecognized command line option: %s\n",argv[i]);
	}
  }

  printf("Mindepth: 2^%d=%d, Maxdepth: 2^%d=%d, Iterations: %d, Solve-Sweeps: %d\n",
		 mindepth,1<<mindepth,logmaxdepth,maxdepth,itmax,(nosolve) ? 0 : solve_sweeps);
}


main(argc,argv)
	 int argc;
	 char *argv[];
{
  double maxerr,avgerr,err;
  node *n;
  int i,p,count,nfs,iter,converge;
  unsigned u,uu;
  chunk *genarg;

  parsecommandline(argc,argv);

  nhead=NULL; thead=NULL;

  if(loadfile!=NULL) {
	FILE *f;
	
	f = fopen(loadfile,"r");
	if(f==NULL) {
	  fprintf(stderr,"Cannot open file %s to load triangulation\n",loadfile);
	  exit(1);
	}

	printf("Loading triangulation from %s\n",loadfile);
	loadtriangulation(f,&nhead,&thead);

	fclose(f);

  } else {
  
	/* Setup grid and initial values */
	create_grid(&nhead,&thead);
	if(nhead==NULL || thead==NULL) {
	  if(thead==NULL) fprintf(stderr,"No triangles\n");
	  else fprintf(stderr,"No nodes\n");
	  exit(1);
	}
	printf("Grid created. Maximum distortion: %lf\n",maxshift);
  
	smooth(nhead,smoothmax);

	if(savefile!=NULL) {
	  FILE *f;
	  
	  f = fopen(savefile,"w");
	  if(f==NULL) {
		fprintf(stderr,"Cannot open file %s to save triangulation\n",savefile);
	  } else {
	  
		savetriangulation(f,nhead,thead);
	  
		fclose(f);
	  }
	}
  }

  start(nhead,phis);

  if(!nosolve) {

	/* And begin iterations */
	for(iter=0;iter<itmax;iter++) {
	  
	  for(nfs=0;nfs<NV;nfs++) {
		
		if(!(phis[nfs].lsolve)) continue;
		
		coeff(nhead,thead,nfs);
		
		bound(nhead,nfs);
		
		relax(nhead,nfs);
		
		solve(nhead,nfs);
	  }
	  
	  converge=monit(nhead,thead);
	  
	  if(converge) break;
	}
  }
	  
  /* Finally, display grids */

  printf("Smoothing: %d",smoothmax);

  if(!nosolve) {
	/* First determine any error between computed solution and exact solution */
	maxerr=0.0; avgerr=0.0; count=0;
	n=nhead;
	while(n!=NULL) {
	  
	  if(!((n->flags)&FLAG_RealBoundary)) {
		err=n->phi[TEXACT] - n->phi[TCOMP];
		if(err<0.0) err = -err;
		
		if(err>maxerr) maxerr=err;
		avgerr+=err;
		
		count++;
	  }
	  
	  n=n->next;
	}	

	if(count>0) avgerr /= count;
	printf(" AvgErr: %lf of %d  MaxErr: %lf\n",avgerr,count,maxerr);	

  } else printf(" AvgErr and MaxErr unavailable\n");
	
  if(!noout) showgrids(nhead,thead,argc,argv);

  return(0);
}
