/* PUG.H   Parallel Unstructured Grid Control Volume Finite Element Method
   Header file */

/* Includes needed by all files */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <limits.h>
#include <values.h>


/* Some needed constants */

#define VERSION "1.8"

#ifndef True
#define False 0
#define True (!False)
#endif

#ifndef PI
#define PI 3.14159265
#endif
#define Epsilon 0.000000001            /* A small number */
#define Infinity (1.0e32)              /* A really big number */
#define BigNumber 500.0                /* A big number */           

#define PTRAD 4                        /* Radius for drawing vertices */

#define Sarea_Left   1                 /* Return values of sarea function */
#define Sarea_None   0 
#define Sarea_Right  (-1)

#define Intersect_No             0     /* Return values of intersect function */
#define Intersect_ColinearSortof 1
#define Intersect_Colinear       2
#define Intersect_Sortof         3
#define Intersect_Yes            4

enum {                                 /* Meaning of different variables */
  TEXACT = 0,                          /*  (Given) exact solution */
  TCOMP,                               /*  Computed solution */
  NV                                   /*  Total # of variables */
  };


/* Bit masks for flag variable of nodes/triangles */
#define FLAG_Boundary     0x01         /* Node is a boundary node */
#define FLAG_RealBoundary 0x02         /* Node is an original boundary node */
#define FLAG_Triangulated 0x04         /* Node is part of triangulation */
#define FLAG_Mark         0x08         /* Node has been marked for some operation */
#define FLAG_Created      0x10         /* Node created during quadtree construction */
#define FLAG_Global       0x20         /* Object is in or points to global memory */
#define FLAG_Temp         0x40         /* To demarcate temporary nodes */

#define ITMAX 1                        /* Default maximum iterations */
#define SMOOTHMAX 0                    /* Default maximum smoothing iterations */
#define SWEEPSMAX 1000                 /* Default maximin solving sweeps */

typedef unsigned exactdouble;          /* My own floating-point */
#define MAXDEPTH 12                    /* Default maximum depth of recursion */

/* Quadtree flags */
#define QUAD_INQ 0x001                 /* Bit to indicate quad is in the queue */
#define QUAD_BND 0x002                 /* Quad contains the domain boundary */
#define QUAD_INT 0x004                 /* Quad is entirely interior to the domain */
#define QUAD_EXT 0x008                 /* Quad is entirely exterior to the domain */

#define QUAD_TOP 0x010                 /* Quad is on top boundary of main quad */
#define QUAD_BOT 0x020
#define QUAD_LEF 0x040
#define QUAD_RIG 0x080

#define QUAD_ACU 0x100                 /* Quad contains an acute boundary angle */

#define QUAD_EDG (QUAD_TOP|QUAD_BOT|QUAD_LEF|QUAD_RIG)

#define CORNER_UL (0x01)               /* Bits indicating the 4 corners of a quad */
#define CORNER_UR (0x02)
#define CORNER_BL (0x04)
#define CORNER_BR (0x08)

/* Some macros to make life easier */

#define Double2ExactDoubleX(D) ((exactdouble)((((D)-gridsx)/griddim)*maxdepth))
#define Double2ExactDoubleY(D) ((exactdouble)((((D)-gridsy)/griddim)*maxdepth))
#define ExactDouble2DoubleX(E) ((((double)(E))/((double)maxdepth))*griddim+gridsx)
#define ExactDouble2DoubleY(E) ((((double)(E))/((double)maxdepth))*griddim+gridsy)

/* Squaring a value */
#define Sq(Value) ((Value) * (Value))

/* Mapping a coordinate onto the output window */
#define CoordX(Value,MaxX) ((int)((Value)*(MaxX-2))+1)
#define CoordY(Value,MaxY) (MaxY-1-((int)((Value)*(MaxY-2))))

/* Determining which corner a node is in a triangle */
#define WhichCorner(Triang,Node) (((Triang)->c[0].n==(Node)) ? 0 : \
								  (((Triang)->c[1].n==(Node)) ? 1 : 2))

/* (In)Equality for floating point numbers */
#define DoubleEQ(D1,D2) (((D1)>(D2)) ? (D1)-(D2)<=Epsilon : (D2)-(D1)<=Epsilon)
#define DoubleLT(D1,D2) ((D2)-(D1)>(Epsilon/2.0))
#define DoubleGT(D1,D2) ((D1)-(D2)>(Epsilon/2.0))
#define DoubleGTE(D1,D2) ((D1)-(D2)>(-Epsilon/2.0))
#define DoubleLTE(D1,D2) ((D2)-(D1)>(-Epsilon/2.0))

/* Testing a quad in the quadtree to see if it's a leaf node */
#define IsLeaf(Q) ((Q)->ul==NULL)

/* Structure definitions */

struct triangle_type;                  /* Just a pre-declaration */

typedef struct node_type {             /* Structure at each grid node */
  
  double x,y,                          /* Each has (x,y) coords */
         phi[NV],                      /*  and phi */
         ac,                           /*  and self */
         acon,                         /*  and the constant */
         gamma,                        /*  and gamma */
         sc,sp;                        /*  and source terms */

  exactdouble ex,ey;                   /* Exact floating-point version of coords */

  int flags;                           /* Some bits for binary states */
  int processor;                       /* Which processor is on */
  int index;                           /* Array position on that processor */

  /* Triangles are organized as an unordered, doubly-connected linked list
	 around each node. The following pointers form the head & tail */
  struct triangle_type *thead;                     
  struct triangle_type *ttail;

  struct node_type *next;              /* Nodes also kept in a (singly) linked list */
  struct node_type *prev;              /* Not used, except in grid creation */

  void *usn;                           /* Pointer to US version of node */
  
} node;


/* A structure for the array of nodes in global memory when passing
   info between processes */
typedef struct nodeandlock_type {

  node n;                              /* The actual node data */
  short l;                             /* A lock for exclusive work on it */

} nodeandlock;


typedef struct corner_type {           /* Triangles consist of corners */

  node *n;                             /* Node forming this corner */
  nodeandlock *usn;                    /* Node in global memory, with lock */
  double a[2];                         /* and coeffs for its neighbours */

  struct triangle_type *next;          /* For doubly-linked list */
  struct triangle_type *prev;

} corner;


typedef struct triangle_type {         /* Triangle structure */

  corner c[3];                         /* Each triangle has 3 corners */
  int flags;                           /* For some bit-flags */
  int processor;                       /* Which processor worked on this */

  struct triangle_type *next;          /* Triangles also in (singly) linked list */

} triangle;


typedef struct chunk_type {            /* Chunk descriptor */
  
  triangle *th;                        /* Array of triangles in chunk */
  int size;                            /* Size of array */

} chunk;


typedef struct phidescr_type {         /* Parameters of each variable to solve for */

  char *title;                         /* Description of this variable */

  int lsolve,                          /* Boolean: solve for this phi */
      lprint,                          /* Boolean: print this phi */
      nswp;                            /* Number of sweeps in solver */

  double relax;                        /* Relaxation factor */

} phidescr;


/* A structure for retaining alternative lists of nodes than
   the main nodelist */
typedef struct nodelist_ {
  node *n;
  struct nodelist_ *next;
  struct nodelist_ *c1,*c2;
} nodelist;


/* A structure for holding alternative lists of edges than
   the info contained in the main nodelist */
typedef struct edgelist_ {
  node *n1,*n2;
  struct edgelist_ *next;
} edgelist;


/* And a quad-tree structure for building the triangulation */
typedef struct quadtree_ {
  int flags;                           /* Bit flags for state */
  int depth;                           /* Depth of quad in quadtree */
  int size;

  nodelist *nlhead;                    /* Nodes contained within this square */
  edgelist *elhead;                    /* Edges contained within this square */

  node *nul,*nur,*nbr,*nbl;            /* Nodes forming corners */  

  struct quadtree_ *ul,*ur,*bl,*br;    /* Children */
  struct quadtree_ *left,*top,*right,*bottom;  /* Siblings */
  struct quadtree_ *parent;            /* Parent */

  struct quadtree_ *next;              /* For queueing quadtrees */
} quadtree;


/* Global variables */
#ifdef _PUGGLOBALS_

phidescr phis[NV];                     /* Array of phi-descriptors */
triangle *thead;                       /* Head of triangle list */
node *nhead;                           /* Head of node list */
char *savefile;                        /* Options from command line */
char *loadfile;
int solve_sweeps = SWEEPSMAX;          /* Max # sweeps in solver */
int nosolve;                           /* Whether to compute a solution or not */
int scale;                             /* Whether to scale grid to fit input */
int maxdepth = (1<<MAXDEPTH);
int mindepth = 0;                      /* Default minimum depth of quadtree */
double maxshift = 0.0;                 /* Maximum distortion in gridpoints */
double griddim,gridsx,gridsy;          /* Grid size and placement */

#else

extern phidescr phis[];
extern int solve_sweeps;
extern int maxdepth;
extern int mindepth;
extern int scale;
extern double maxshift;
extern double griddim,gridsx,gridsy;

#endif


/* Function prototypes */

/* In PUG.C */

/* In PUGUSR.C */
void start(/* node *nh; phidescr *phis; */);
void gamsor(/* node *nh; int nfs; */);
void bound(/* node *nh; int nfs; */);
int monit(/* node *nh; triangle *th; */);

/* In PUGGRID.C */
void create_grid(/* node **nh; triangle **th; */);

/* In PUGALLOC.C */
node *newnode(/* double x,y; int flags; */);
triangle *make_triangle(/* triangle **,node *,node *,node * */);
void llinsert(/* node *n; triangle *t; */);
void lldelete(/*node *n; triangle *t; */);
void unmake_triangle(/*triangle **th; triangle *tdel; */);

/* In PUGDEL.C */
void delaunay(/* node *nh; triangle **th; */);

/* In PUGNODELIST.C */
nodelist *newnodelist(/* node *n; */);
node *extractnode(/* nodelist **nl; exactdouble ex,ey; */);
node *findnode(/* nodelist *nlh; double x,y; */);
void freenodelist(/* nodelist *nl; */);
void divynodelist(/* nodelist *nlhead,**nlul,**nlur,**nlbr,**nlbl; 
					 exactdouble sx,sy,ex,ey,mx,my; */);

/* In PUGEDGELIST.C */
int edgeintersects(/* edgelist *e; exactdouble sx,sy,ex,ey; */);
int edgesokinquad(/* quadtree *q; edgelist *ehead; */);
edgelist *buildedgelist(/* node *nhead; */);
edgelist *newedgelist(/* node *n1,*n2; */);
void destroyedgelist(/* edgelist *ehead; */);
void traceboundary(/* quadtree *q; */);
void divyedgelist(/* quadtree *q; edgelist **elul,**elur,**elbr,**elbl;
					 exactdouble sx,sy,ex,ey,midx,midy; */);
edgelist *closesthedge(/* exactdouble sx,y,ex,nearestx; edgelist *elhead; */);
edgelist *closestvedge(/* exactdouble x,sy,ey,nearesty; edgelist *elhead; */);


/* In PUGQUAD.C */
void triangulate_byquadtree(/* node **nh; triangle **th; */);

/* In PUGTRI0.C */
void triangulate_quad(/* triangle **th; quadtree *q; node *nt,*nr,*nb,*nl; */);
void triangulate_quad0(/* triangle **th; quadtree *q; node *nt,*nr,*nb,*nl; */);

/* In PUGTRI1.C */
void triangulate_quad1(/* triangle **th; quadtree *q; node *nt,*nr,*nb,*nl; */);

/* In PUGTRI2.C */
void triangulate_quad2(/* triangle **th; quadtree *q; node *nt,*nr,*nb,*nl; */);

/* In PUGREDO.C */
void forceedges(/*node *nh; triangle **th; */);
void deletealltriangles(/*node *nh; triangle **th; */);

/* In PUGBFS.C */
void removeexternaltriangles(/* triangle **th; edgelist *elhead; */);

/* In PUGQUADOP.C */
void rotateclockwise(/* node **nul,**nur,**nbr,**nbl,**nt,**nr,**nb,**nl; */);
void rotatecounterclockwise(/* node **nul,**nur,**nbr,**nbl,**nt,**nr,**nb,**nl; */);
void reflectvertically(/* node **nul,**nur,**nbr,**nbl,**nt,**nr,**nb,**nl; */);
void reflecthorizontally(/* node **nul,**nur,**nbr,**nbl,**nt,**nr,**nb,**nl; */);

/* In PUGPURGE.C */
void purgetriangles(/* node *nh; triangle **th; */);

/* In PUGREORDER.C */
void reordernodes(/* node **nh; */);
node *shiftnodes(/* node **nh; int flagmask; */);

/* In PUGSMOOTH.C */
void smooth(/* node *nh; int maxiter; */);

/* In PUGFILE.C */
int savetriangulation(/* FILE *f; node *nh; triangle *th; */);
int loadtriangulation(/* FILE *f; node **nh; triangle **th; */);

/* In PUGCOEFF.C */
void coeffinit(/* node *nh; triangle *th; */);
int coeffworker(/* chunk *clist; */);

/* In PUGSOLVE.C */
void solve(/* node *nh; int nfs; */);

/* In PUGOUT.C */
void showgrids(/* node *nh; triangle *th; int argc; char *argv[]; */);
void make2d(/* double x,y,z,*a,*b; */);
void rotateyz(/* double a; */);
void rotateyx(/* double a; */);
void rotatexz(/* double a; */);
void flatten();

/* In PUGX.C */
void showxgrids(/* int argc; char *argv[]; node *nh; triangle *th; */);

/* In SAREA.C */
int sarea(/* node *,node *,node * */);
int intersectionpoint(/* node *s0,*e0,*s1,*e1; double *x1,*yi; */);
int insidesquare(/* node *p,*c1,*c2,*c3,*c4; */);

