/* PUGALLOC.C   Creation and deletion of nodes and triangles */


#include "pug.h"


/* Allocates a new node and sets fields to default or given values.
   Returns ptr to new node. Exits if out of memory */
node *newnode(x,y,flags)
	 double x,y;
	 int flags;
{
  node *n;

  n=(node *)malloc(sizeof(node));
  if(n==NULL) {
	fprintf(stderr,"Out of memory. Sorry.\n");
	exit(1);
  }

  n->x=x;
  n->y=y;
  n->flags=flags;

  n->thead=NULL;
  n->ttail=NULL;
  n->next=NULL;
  n->prev=NULL;

  return(n);
}


/* Actually constructs the triangle between (n1,n2,n3) */
triangle *make_triangle(th,n1,n2,n3)
	 triangle **th;
	 node *n1;
	 node *n2;
	 node *n3;
{
  node *n;
  triangle *t;

  t=(triangle *)malloc(sizeof(triangle));
  if(t==NULL) {
	fprintf(stderr,"Out of memory. Sorry.\n");
	exit(1);
  }

  /* Make sure vertices numbered locally in counterclockwise area */
  if(sarea(n1,n2,n3)==Sarea_Right) {
	n=n2;
	n2=n3;
	n3=n;
  }

  t->c[0].n = n1; t->c[0].next=NULL; t->c[0].prev=NULL;
  t->c[1].n = n2; t->c[1].next=NULL; t->c[1].prev=NULL;
  t->c[2].n = n3; t->c[2].next=NULL; t->c[2].prev=NULL;

  t->flags=0;
  t->processor =0;

  /* And insert t into the linked lists of each node */
  llinsert(n1,t);
  llinsert(n2,t);
  llinsert(n3,t);

  /* Tack onto linked list of triangles */
  t->next= *th;
  *th=t;

  return(t);
}


/* -------------------------------------------------------------------*/
/* The companion function to make_triangle -- destroys the given triangle
   and 'undraws' it on the screen */
void unmake_triangle(th,tdel)
	 triangle **th;
	 triangle *tdel;
{
  triangle *tprev;

  lldelete(tdel->c[0].n,tdel);
  lldelete(tdel->c[1].n,tdel);
  lldelete(tdel->c[2].n,tdel);

  if(*th==tdel) *th=tdel->next;
  else {

	/* Locate triangle prior to tdel in linked list */
	tprev = *th;
	while(tprev!=NULL && tprev->next!=tdel) tprev=tprev->next;

	if(tprev!=NULL) 
	  tprev->next=tdel->next;
	else
	  fprintf(stderr,"Unable to locate triangle to delete...\n");
  }

  free(tdel);
}


/* Insert a given triangle into the linked-list that surrounds each node 
   is incident */
void llinsert(n,t)
	 node *n;
	 triangle *t;
{
  triangle *tt;
  int tcorn,ttcorn;

  tcorn=WhichCorner(t,n);
  t->c[tcorn].next=NULL;
  t->c[tcorn].prev=NULL;

  if(n->thead==NULL) {       /* Only item in linked list */

	n->thead=t;
	n->ttail=t;
	
  } else {

	/* else insert at the front of the linked list */
	tt=n->thead;
	ttcorn=WhichCorner(tt,n);
	
	tt->c[ttcorn].prev=t;
	t->c[tcorn].next=tt;
	
	n->thead=t;
  }
}


/* Remove a given triangle from the linked list around each node */
void lldelete(n,t)
	 node *n;
	 triangle *t;
{
  int tcorn;
  triangle *tnext,*tprev;

  tcorn=WhichCorner(t,n);

  if(n->thead==n->ttail) {          /* Only item left in linked list */

	n->thead=NULL;
	n->ttail=NULL;

  } else {

	tnext=t->c[tcorn].next;
	tprev=t->c[tcorn].prev;

	if(tnext==NULL) {               /* Last item in linked list */

	  tprev->c[WhichCorner(tprev,n)].next=NULL;
	  n->ttail=tprev;
	  
	} else {

	  if(tprev==NULL) {             /* First item in linked list */
		
		tnext->c[WhichCorner(tnext,n)].prev=NULL;
		n->thead=tnext;

	  } else {                      /* None of the above */

		tnext->c[WhichCorner(tnext,n)].prev=tprev;
		tprev->c[WhichCorner(tprev,n)].next=tnext;
	  }
	}
  }

  t->c[tcorn].next=NULL;
  t->c[tcorn].prev=NULL;
}
