/* PUGDEL.C  Computes Delaunay triangulation from a linked list
             of nodes. */

#include "pug.h"


static node *centrenode;



void circletransform(/* node *,node *,node *,double *,double *,double * */);
double distance2(/* double,double,node * */);
void watson(/* node *nh; triangle **th; */);
int marktriangles(/* triangle *th; node *n; */);
int removeduplicates(/* node **nh; int count; */);
int radialcompare(/* node **n1,**n2; */);
node **sortnodes(/* triangle *th; node *ncentre; int *c; */);
void insertpoint(/* triangle **th; node *nnew; int count; */);
void removemarked(/* triangle **th; */);
void markadjacentnodes(/* node *n; int flag; */);
static void removenode(/* triangle **th; node *n; */);



/* Construct Delaunay triangulation via Watson's algorithm */
void delaunay(nh,th)
	 node *nh;
	 triangle **th;
{
  node *n1,*n2,*n3;

  /* Reset flags in each node */
  n1=nh;
  while(n1!=NULL) {
	
	n1->flags &= ~(FLAG_Triangulated|FLAG_Boundary);
	n1=n1->next;
  }

  /* Form initial triangle that is guaranteed to contain all
	 subsequent points */
  n1=newnode(-10.0,0.0,FLAG_Boundary);
  n2=newnode(0.0,10.0,FLAG_Boundary);
  n3=newnode(10.0,0.0,FLAG_Boundary);

  n1->next=n2;
  n2->next=n3;
  n3->next=nh;

  make_triangle(th,n1,n2,n3);

  /* Then start the actual triangulation */
  watson(nh,th);

  /* Ensure boundary edges are included in the triangulation */
  forceedges(nh,th);

  /* Every node attached to the n1,n2,n3 must be a boundary
	 node, so we mark them as such */
  markadjacentnodes(n1,FLAG_Boundary);
  markadjacentnodes(n2,FLAG_Boundary);
  markadjacentnodes(n3,FLAG_Boundary);

  /* Now to delete unnecessary triangles attached to 3 initial nodes */
  removenode(th,n1);
  removenode(th,n2);
  removenode(th,n3);

  /* And delete original bounding triangle nodes */
  free(n1);
  free(n2);
  free(n3);

  /* And finally delete any triangles that lie outside the 
	 domain */
  purgetriangles(nh,th);
}


/* Generates the equation of a circle in the form
   (x+cx)^2 + (y+cy)^2 = r^2
   from 3 nodes (not colinear or 2 coincident) on the 
	circumference. Note that the centre of the circle
	would be at (-cx,-cy) */
void circletransform(n1,n2,n3,cx,cy,r2)
	 node *n1;
	 node *n2;
	 node *n3;
	 double *cx;
	 double *cy;
	 double *r2;
{
  double x1,x2,x3,y1,y2,y3,denom;

  x1 = n1->x; x2 = n2->x; x3 = n3->x;
  y1 = n1->y; y2 = n2->y; y3 = n3->y;

  if(y2==y3) {

	x1 = n3->x; x2 = n1->x; x3 = n2->x;
	y1 = n3->y; y2 = n1->y; y3 = n2->y;
  }	

  if(y2==y3) {
	
	fprintf(stderr,"Warning: 3 colinear points defining a circle");
	fprintf(stderr,"(%g,%g),(%g,%g),(%g,%g)\n",x1,y1,x2,y2,x3,y3);

	*cx = Infinity;
	*cy = Infinity;
	*r2  = Infinity;

	return;
  }
   
  denom = 2.0 * ((x1-x3) + ((y1-y3)*(x3-x2)) / (y2-y3));

  if(denom==0.0) {
	
	fprintf(stderr,"Warning: 3 colinear points defining a circle\n");

	*cx = Infinity;
	*cy = Infinity;
	*r2  = Infinity;

	return;
  }

  *cx = Sq(x3)-Sq(x1) + Sq(y3)-Sq(y1) -
	    ((y1-y3)*((Sq(y3)-Sq(y2)) + (Sq(x3)-Sq(x2)))) / (y2-y3);

  *cx = *cx / denom;

  *cy = (Sq(y3)-Sq(y2) + Sq(x3+ *cx) - Sq(x2+ *cx)) / (2.0 * (y2-y3));

  *r2 = Sq(x3 + *cx) + Sq(y3 + *cy);
}


/* returns the Euclidean distance^2 from (cx,cy) to n */
double distance2(cx,cy,n)
	 double cx;
	 double cy;
	 node *n;
{

  return(Sq(n->x-cx)+Sq(n->y-cy));
}


/* Implements Watson's algorithm for Delaunay triangulation */
void watson(nh,th)
	 node *nh;
	 triangle **th;
{
  int c;
  node *n;
  triangle *t;

  n=nh;
  while(n!=NULL) {

	/* Determine which triangles will need to be deleted */
	c=marktriangles(*th,n);

	/* Insert new triangles */
	insertpoint(th,n,c);

	/* Delete unwanted triangles */
	removemarked(th);

	n=n->next;
  }
}

/* Locates all triangles which contain n within their circumcircle
   and marks them. This is a naive linear scan of the triangle list.
   Returns the number of marked triangles */
int marktriangles(th,n)
	 triangle *th;
	 node *n;
{
  int count;
  double r2,cx,cy,d2;
  triangle *t;

  t=th;
  count=0;
  while(t!=NULL) {

	circletransform(t->c[0].n,t->c[1].n,t->c[2].n,&cx,&cy,&r2);
	cx = -cx; cy = -cy;

	d2=distance2(cx,cy,n);

	if(d2<r2-Epsilon) {
	  t->flags |= FLAG_Mark;
	  count++;
	}

	t=t->next;
  }

  return(count);
}


/* Deletes all adjacent duplicate entries in the array nh containing
   c elements. Returns the number of elements left in the set */
int removeduplicates(nh,count)
	 node **nh;
	 int count;
{
  int c;
  node **nbase,**n;
  
  if(count==0) return(0);

  nbase=nh;
  n=nh+1;

  c=1; count--;
  while(count>0) {

	while(*n==*nbase && count>0) {
	  
	  count--;
	  n++;
	}
	if(count<=0) break;

	nbase++;
	*nbase = *n;
	n++;
	c++;
	count--;
  }

  return(c);
}

#define Quadrant(Eks,Why) (((Eks)>=0.0) ? (((Why)>=0.0) ? 1 : 4) : \
						   (((Why)>=0.0) ? 2 : 3))


/* Comparison function for quicksort */
int radialcompare(n1,n2)
	 node **n1,**n2;
{
/*  int i;
  double a1,a2;

  if(*n1==*n2) return(0);

  a1=atan2(((*n1)->y-centrenode->y),((*n1)->x-centrenode->x));
  a2=atan2(((*n2)->y-centrenode->y),((*n2)->x-centrenode->x));

  return((a1<a2) ? -1 : ((a1>a2) ? 1 : 0));
*/
  int q1,q2;
  node nn1,nn2,cn;

  /*First transform n1 and n2 so they're relative to centrenode*/
  nn1.x = (*n1)->x - centrenode->x;
  nn1.y = (*n1)->y - centrenode->y;
  nn2.x = (*n2)->x - centrenode->x;
  nn2.y = (*n2)->y - centrenode->y;

  /*Then check which quadrant they're in*/
  q1 = Quadrant(nn1.x,nn1.y);
  q2 = Quadrant(nn2.x,nn2.y);

  if(q1!=q2) return(q1-q2);

  cn.x = 0.0; cn.y = 0.0;

  return(sarea(&nn1,&cn,&nn2));
}


/* For all tcount marked triangles, dumps the associated nodes into
   an array, and then sorts them radially about ncentre. 
   Sorted nodes are returned as a (new) array, which must be free()-ed. */
node **sortnodes(th,ncentre,count)
	 triangle *th;
	 node *ncentre;
	 int *count;
{
  int ncount;
  triangle *t;
  node **snh,**n,**nbase;

  /* There are at most 3 nodes for every triangle, so by
	 allocating space for 3*tcount node pointers, there will surely
	 be enough room */
  snh=(node **)malloc(sizeof(node *) * (*count) * 3);
  if(snh==NULL) {
	fprintf(stderr,"Out of memory. Sorry.\n");
	exit(1);
  }

  /* Copy nodes adjacent to marked triangles into allocate array */
  n=snh;
  t=th;
  ncount = 0;
  while(t!=NULL) {

	if((t->flags)&FLAG_Mark) {

	  *n = t->c[0].n; n++; ncount++;
	  *n = t->c[1].n; n++; ncount++;
	  *n = t->c[2].n; n++; ncount++;
	}

	t=t->next;
  }

  centrenode=ncentre;

  /* Now they're all loaded into a single array, sort 'em */
  qsort(snh,ncount,sizeof(node *),radialcompare);

  *count = ncount;

  return(snh);
}


/* Manages the insertion of a new point into the existing triangulation, 
   given that all the triangles affected by insertion have already been
   marked. Creates the appropriate new triangles, but does not
   delete the old ones or link in the new point */
void insertpoint(th,nnew,tcount)
	 triangle **th;
	 node *nnew;
	 int tcount;
{
  int i,ncount;
  node **snh;

  /* Sort marked nodes by angle around n */
  ncount = tcount;
  snh=sortnodes(*th,nnew,&ncount);

  /* Delete duplicate copies of the same node */
  ncount=removeduplicates(snh,ncount);

  /* Add in all the new triangles */
  for(i=0;i<ncount;i++) 
	make_triangle(th,nnew,snh[i],snh[(i+1)%ncount]);
	
  /* Delete allocated array */
  free(snh);
}


/* Removes all marked triangles from the triangle list and free()s them */
void removemarked(th)
	 triangle **th;
{
  triangle *t,*tlast;

  tlast=NULL;
  t = *th;
  while(t!=NULL) {

	if((t->flags)&FLAG_Mark) {

	  if(tlast==NULL) *th=t->next;
	  else tlast->next=t->next;

	  lldelete(t->c[0].n,t);
	  lldelete(t->c[1].n,t);
	  lldelete(t->c[2].n,t);

	  free(t);

	  if(tlast!=NULL) t=tlast->next;
	  else t = *th;

	} else {

	  tlast=t;
	  t=t->next;
	}
  }

}


/* ORs the flags field of all adjacent nodes to the given node with the given
   flag value */
void markadjacentnodes(n,flag)
	 node *n;
	 int flag;
{
  triangle *t;

  t=n->thead;
  while(t!=NULL) {
	
	(t->c[0].n)->flags |= flag;
	(t->c[1].n)->flags |= flag;
	(t->c[2].n)->flags |= flag;

	t=t->c[WhichCorner(t,n)].next;
  }
}


/* Deletes all triangles linked to the given node */
static void removenode(th,n)
	 triangle **th;
	 node *n;
{

  while(n->thead!=NULL) 
	unmake_triangle(th,n->thead);
}
