/* PUGEDGELIST.C  Module for handling and building list of edges */


#include "pug.h"

static void splitedgelist(/* quadtree *q; edgelist *e; double x,y;
							 exactdouble ex,ey; */);


edgelist *newedgelist(n1,n2)
	 node *n1,*n2;
{
  edgelist *e;

  e = (edgelist *)malloc(sizeof(edgelist));
  if(e==NULL) {
	fprintf(stderr,"Out of memory. Sorry.\n");
	exit(1);
  }
  
  e->n1 = n1; e->n2 = n2;
  e->next = NULL;

  if(n1->x==n2->x && n1->y==n2->y) {
	fprintf(stderr,"Warning: Edge of zero length just created\n");
  }

  return(e);
}


/* Extracts the list of edges from the given list of nodes */
edgelist *buildedgelist(nhead)
	 node *nhead;
{
  node *n,*nnext;
  edgelist *ehead,*etail,*e;

  if(nhead==NULL || !((nhead->flags)&FLAG_RealBoundary))
	return(NULL);

  ehead = etail = NULL;
  n=nhead;
  while(n!=NULL && ((n->flags)&FLAG_RealBoundary)) {

	nnext = (n->next==NULL || !((n->next->flags)&FLAG_RealBoundary)) ? nhead : n->next;
	e = newedgelist(n,nnext);
	
	if(ehead==NULL) ehead = e;
	else etail->next = e;
	etail = e;

	n = n->next;
  }

  return(ehead);
}

int showelist(ehead)
	 edgelist *ehead;
{
  edgelist *e;

  e = ehead;
  while(e!=NULL) {
	if(e->n1==NULL || e->n2==NULL)
	  return(1);
	printf("Edge: (%g,%g)-(%g,%g)\n",e->n1->x,e->n1->y,e->n2->x,e->n2->y);
	e = e->next;
  }
  printf("All edges shown\n");

  return(0);
}


void destroyedgelist(ehead)
	 edgelist *ehead;
{
  edgelist *e,*enext;

  e = ehead;
  while(e!=NULL) {
	enext = e->next;
	free(e);
	e = enext;
  }
}


/* Answers True if the given edge is strictly contained within
   the given boundaries -- ie, both endpoints lie on or within
   the given square. Colinear edges contained within the 
   boundary are considered strictly within */
int strictlywithin(e,sx,sy,ex,ey)
	 edgelist *e;
	 exactdouble sx,sy,ex,ey;
{
  double x1,y1,x2,y2;
  
  /* A line is strictly contained with a square if the two
	 endpoints are */

  x1 = ExactDouble2DoubleX(sx); y1 = ExactDouble2DoubleY(sy);
  x2 = ExactDouble2DoubleX(ex); y2 = ExactDouble2DoubleY(ey);

  if(DoubleGTE(e->n1->x,x1) && DoubleLTE(e->n1->x,x2) && 
	 DoubleGTE(e->n1->y,y1) && DoubleLTE(e->n1->y,y2) &&
	 DoubleGTE(e->n2->x,x1) && DoubleLTE(e->n2->x,x2) && 
	 DoubleGTE(e->n2->y,y1) && DoubleLTE(e->n2->y,y2)) {
	return(True);
  }

  return(False);
}


/* Splits the given edge around the given point. If a node for
   the given coordinates does not exist, then one will be allocated
   and added to the quad's nodelist */
static void splitedgelist(q,e,x,y,ex,ey) 
	 quadtree *q;
	 edgelist *e;
	 double x,y;
	 exactdouble ex,ey;
{
  node *n;
  edgelist *etemp;

  n = findnode(q->nlhead,x,y);
  if(n==NULL) {
	nodelist *nl;
	
	/* If no node existed in q's nodelist, then create one and
	   add it to q's nodelist */
	n = newnode(x,y,FLAG_Created);
	n->ex = ex; n->ey = ey;

	nl = newnodelist(n);
	nl->next = q->nlhead;
	q->nlhead = nl;

  } else {

	if(n->next!=NULL) n->next->prev = n->prev;
	if(n->prev!=NULL) n->prev->next = n->next;
  }

  n->flags |= FLAG_RealBoundary;
  
  if(((e->n1->flags)&FLAG_RealBoundary) && ((e->n2->flags)&FLAG_RealBoundary)) {
	/* Also patch it into the node list in the appropriate spot */

	n->next = e->n1->next;
	if(n->next!=NULL) n->next->prev = n;
	n->prev = e->n1;
	e->n1->next = n;
  }

  /* Split the edge into 2 pieces, tack them onto the 
	 front of the list of edges and continue */
  etemp = newedgelist(n,e->n2);
  etemp->next = e->next;
  e->n2 = n;
  e->next = etemp;
}


void divyedgelist(q,eltl,eltr,elbr,elbl,sx,sy,ex,ey,midx,midy)
	 quadtree *q;
	 edgelist **eltl,**eltr,**elbr,**elbl;
	 exactdouble sx,sy,ex,ey,midx,midy;
{
  double x1,y1,x2,y2;
  edgelist *e,*enext;
  
  /* For each edge in our input list, we need to 
	 determine which child squares it would intersect,
	 and pass it down to the appropriate children.
	 If the same edge intersects more than one child
	 square, then a node or nodes is/are introduced to
	 split the edge and each child gets a piece.
	 Because we ensure this condition for each child,
	 we can assume that every edge is strictly contained
	 within the current quad. As well, it is assumed that
	 this function is called before the nodes for the
	 children are removed from the quad's nlhead. */
  
  *eltl = *eltr = *elbr = *elbl = NULL;

  e = q->elhead;
  while(e!=NULL) {
	enext = e->next;

	if(!strictlywithin(e,sx,sy,ex,ey)) {
	  fprintf(stderr,"Buggery! Edge (%g,%g)-(%g,%g)\n",e->n1->x,e->n1->y,
			  e->n2->x,e->n2->y);
	  fprintf(stderr," doesn't all lie within (%g,%g)-(%g,%g)\n",
			  ExactDouble2DoubleX(sx),ExactDouble2DoubleY(sy),
			  ExactDouble2DoubleX(ex),ExactDouble2DoubleY(ey));	 
	}

	/* A Strict inclusion means that the edge won't be
	   split, so it only has to be dumped in the appropriate list */
	if(strictlywithin(e,sx,midy,midx,ey)) {
	  e->next = *eltl;
	  *eltl = e;	  
	} else if(strictlywithin(e,midx,midy,ex,ey)) {
	  e->next = *eltr;
	  *eltr = e;
	} else if(strictlywithin(e,sx,sy,midx,midy)) {
	  e->next = *elbl;
	  *elbl = e;
	} else if(strictlywithin(e,midx,sy,ex,midy)) {
	  e->next = *elbr;
	  *elbr = e;
	} else {
	  double cx,cy;

	  /* If no strict inclusions, then the edge must intersect
		 at least 2 child squares. This means it must intersect
		 at least one of the two 'dividing-lines' that define 
		 the children */
	  
	  /* Check to see if its colinear with the one of the dividing lines */
	  cx = ExactDouble2DoubleX(midx); cy = ExactDouble2DoubleY(midy);
	  if((DoubleEQ(e->n1->x,cx) && DoubleEQ(e->n2->x,cx)) ||
		 (DoubleEQ(e->n1->y,cy) && DoubleEQ(e->n2->y,cy))) {

		/* An edge colinear with a dividing line must've crossed
		   the center-point in order to not have been absorbed by
		   one of the tests above. Thus, we only need to split it
		   at the center-point, and then reconsider the two pieces */

		splitedgelist(q,e,cx,cy,midx,midy);

	  } else {
		int i;
		double x,y;
		node n1,n2,n3,n4;

		/* The edge is not colinear with a dividing line, so it must
		   properly intersect one of the dividing lines */

		n1.x = cx; n1.y = ExactDouble2DoubleY(sy);
		n2.x = cx; n2.y = ExactDouble2DoubleY(ey);
		n3.x = ExactDouble2DoubleX(sx); n3.y = cy;
		n4.x = ExactDouble2DoubleX(ex); n4.y = cy;

		i = intersectionpoint(&n1,&n2,e->n1,e->n2,&x,&y);
		if(i && (!DoubleEQ(x,e->n1->x) || !DoubleEQ(y,e->n1->y)) && 
		   (!DoubleEQ(x,e->n2->x) || !DoubleEQ(y,e->n2->y))) {
		  splitedgelist(q,e,x,y,Double2ExactDoubleX(x),Double2ExactDoubleY(y));
		} else if(intersectionpoint(&n3,&n4,e->n1,e->n2,&x,&y)) {
		  splitedgelist(q,e,x,y,Double2ExactDoubleX(x),Double2ExactDoubleY(y));
		} else {
		  fprintf(stderr,"Unanticipated case in edgelist splitting...\n");
		  fprintf(stderr," Edge (%g,%g)-(%g,%g) ",e->n1->x,e->n1->y,
				  e->n2->x,e->n2->y);
		  fprintf(stderr," within (%g,%g)-(%g,%g)\n",
				  ExactDouble2DoubleX(sx),ExactDouble2DoubleY(sy),
				  ExactDouble2DoubleX(ex),ExactDouble2DoubleY(ey));	 
		  exit(1);
		}
	  }
	  continue;
	}
	
	e = enext;
  }
}


/* Assuming that at least one child square does not contain the boundary,
   and that at least one does, this routine determines which child quads
   are to be considered inside or outside the domain 
void traceboundary(q)
	 quadtree *q;
{
  int i;
  quadtree *qc;
  edgelist *c_ul1,*c_ur1,*c_bl1,*c_br1;
  edgelist *c_ul2,*c_ur2,*c_bl2,*c_br2;
  edgelist *cc_ul,*cc_ur,*cc_bl,*cc_br;

   Find the most clockwise and most counterclockwise intersections
	 of edges with the outer boundary of each child quad 

  qc = q->ul;
  if(qc->elhead!=NULL) {
	c_ul1 = closesthedge(qc->nul->ex,qc->nul->ey,qc->nur->ex,qc->nur->ex,qc->elhead);
	c_ul2 = closestvedge(qc->nbl->ex,qc->nbl->ey,qc->nul->ey,qc->nul->ey,qc->elhead);

	cc_ul = closestvedge(qc->nbl->ex,qc->nbl->ey,qc->nul->ey,qc->nbl->ey,qc->elhead);   
	if(cc_ul==NULL) 
	  cc_ul = closesthedge(qc->nul->ex,qc->nul->ey,qc->nur->ex,qc->nul->ex,qc->elhead);

  } else c_ul = cc_ul = NULL;

  qc = q->ur;
  if(qc->elhead!=NULL) {
	c_ur1 = closestvedge(qc->nur->ex,qc->nur->ey,qc->nbr->ey,qc->nbr->ey,qc->elhead);
	c_ur2 = closesthedge(qc->nul->ex,qc->nul->ey,qc->nur->ex,qc->nur->ex,qc->elhead);

	cc_ur = closesthedge(qc->nul->ex,qc->nul->ey,qc->nur->ex,qc->nul->ex,qc->elhead);
	if(cc_ur==NULL) 
	  cc_ur = closestvedge(qc->nbr->ex,qc->nbr->ey,qc->nur->ey,qc->nur->ey,qc->elhead);

  } else c_ur = cc_ur = NULL;

  qc = q->br;
  if(qc->elhead!=NULL) {
	c_br1 = closesthedge(qc->nbl->ex,qc->nbl->ey,qc->nbr->ex,qc->nbl->ex,qc->elhead);
	c_br2 = closestvedge(qc->nbr->ex,qc->nbr->ey,qc->nur->ey,qc->nbr->ey,qc->elhead);

	cc_br = closestvedge(qc->nbr->ex,qc->nbr->ey,qc->nur->ey,qc->nur->ey,qc->elhead);   
	if(cc_br==NULL) 
	  cc_br = closesthedge(qc->nbl->ex,qc->nbl->ey,qc->nbr->ex,qc->nbr->ex,qc->elhead);

  } else c_br = cc_br = NULL;

  qc = q->bl;
  if(qc->elhead!=NULL) {
	c_bl1 = closestvedge(qc->nbl->ex,qc->nbl->ey,qc->nul->ey,qc->nul->ey,qc->elhead);
	c_bl2 = closesthedge(qc->nbl->ex,qc->nbl->ey,qc->nbr->ex,qc->nbl->ex,qc->elhead);

	cc_bl = closesthedge(qc->nbl->ex,qc->nbl->ey,qc->nbr->ex,qc->nbr->ex,qc->elhead);
	if(cc_bl==NULL) 
	  cc_bl = closestvedge(qc->nbl->ex,qc->nbl->ey,qc->nul->ey,qc->nbl->ey,qc->elhead);

  } else c_bl = cc_bl = NULL;


  qc = q->ul;
  if(!((qc->flags)&QUAD_BND)) {

	if(c_bl1!=NULL && (i = sarea(c_bl1->n1,c_bl1->n2,qc->nul))!=Sarea_None) {
	  if(i==Sarea_Left)
		qc->flags |= QUAD_INT;
	  else 
		qc->flags |= QUAD_EXT;

	} else if(c_bl2!=NULL) {
	  i = sarea(c_bl2->n1,c_bl2->n2,qc->nul);
	  if(i==Sarea_Left || (i==Sarea_None && sarea(c_bl2->n1,c_bl2->n2,qc->nbl)==Sarea_Left))
		qc->flags |= QUAD_INT;
	  else 
		qc->flags |= QUAD_EXT;

	} else if(c_br1!=NULL) {
	  i = sarea(c_br1->n1,c_br1->n2,qc->nul);
	  if(i==Sarea_Left || (i==Sarea_None && sarea(c_br1->n1,c_br1->n2,qc->nbl)==Sarea_Left))
		qc->flags |= QUAD_INT;
	  else
		qc->flags |= QUAD_EXT;
	  
	} else if(c_br2!=NULL) {
	  i = sarea(c_br2->n1,c_br2->n2,qc->nul);
	  if(i==Sarea_Left || (i==Sarea_None && sarea(c_br1->n1,c_br1->n2,qc->nbl)==Sarea_Left))
		qc->flags |= QUAD_INT;
	  else
		qc->flags |= QUAD_EXT;

	} else if(c_ur1!=NULL) {
	}	
  }
}*/


/* Locates the edgelist that intersects the given horizontal line closest
   to the given x-coordinate within the given edgelist */
edgelist *closesthedge(sx,y,ex,nearestx,elhead)
	 exactdouble sx,y,ex,nearestx;
	 edgelist *elhead;
{
  exactdouble closestedist;
  edgelist *e,*closeste;

  e = elhead;
  closeste = NULL;
  closestedist = 0;
  while(e!=NULL) {

	if(e->n1->ey==y && (closeste==NULL || abs(e->n1->ex-nearestx)<closestedist)) {
	  closestedist = abs(e->n1->ex-nearestx);
	  closeste = e;
	} 
	if(e->n2->ey==y && (closeste==NULL || abs(e->n2->ex-nearestx)<closestedist)) {
	  closestedist = abs(e->n2->ex-nearestx);
	  closeste = e;
	}

	e = e->next;
  }

  return(closeste);
}



/* Locates the edgelist that intersects the given vertical line closest
   to the given y-coordinate within the given edgelist */
edgelist *closestvedge(x,sy,ey,nearesty,elhead)
	 exactdouble x,sy,ey,nearesty;
	 edgelist *elhead;
{
  exactdouble closestedist;
  edgelist *e,*closeste;

  e = elhead;
  closeste = NULL;
  closestedist = 0;
  while(e!=NULL) {

	if(e->n1->ex==x && (closeste==NULL || abs(e->n1->ey-nearesty)<closestedist)) {
	  closestedist = abs(e->n1->ey-nearesty);
	  closeste = e;
	} 
	if(e->n2->ex==x && (closeste==NULL || abs(e->n2->ey-nearesty)<closestedist)) {
	  closestedist = abs(e->n2->ey-nearesty);
	  closeste = e;
	}

	e = e->next;
  }

  return(closeste);
}


/* Each boundary edge
node *determineboundary(nhead,nlocal)
	 node *nhead,*nlocal;
{
  node *n1,*n2,*n,*nnext;

  n1 = nhead;
  while(n1!=NULL) {

	n2 = (n1->next==NULL) ? nhead : n1->next;

	n = n1;
	while(n!=n2) {
	  nnext = colinearedge(n,n2);	  
	  if(nnext!=n2) {


		if(nnext->prev!=NULL) nnext->prev->next = nnext->next;
		else nlocal = nnext->next;
		if(nnext->next!=NULL) nnext->next->prev = nnext->prev;


		nnext->next = n->next;
		nnext->prev = n;
		if(n->next!=NULL) n->next->prev = nnext;
		n->next = nnext;

		nnext->flags |= FLAG_RealBoundary;
	  }

	  n = nnext;
	}

	n1 = n2;
  }

  return(nlocal);
}


static node *colinearedge(n1,n2)
	 node *n1,*n2;
{
  int tcorn,
  triangle *t;
  node *n;

  t = n1->thead;
  while(t!=NULL) {
	tcorn = WhichCorner(t,n1);

	n = t->c[(tcorn+1)%3].n;
	if(sarea(n1,n,n2)==Sarea_None) 
	  return(n);
	n = t->c[(tcorn+2)%3].n;
	if(sarea(n1,n,n2)==Sarea_None) 
	  return(n);

	t=t->c[tcorn].next;	
  }
}
*/

/*
static void fillquadqueue(q)
	 quadtree *q;
{
  if(IsLeaf(q) && ((q->flags)&QUAD_EDG)) {
	enqueue_quad(q);
  } else {
	fillquadqueue(q->ul);
	fillquadqueue(q->ur);
	fillquadqueue(q->br);
	fillquadqueue(q->bl);
  }
}


void cutoutdomain(qroot)
	 quadtree *qroot;
{
  quadtree *q;

   Build up a queue just of the leaf nodes along the outermost
	 quad boundary. If any of these is not an authentic domain-boundary
	 quad, then it must be an exterior quad 
  fillquadqueue(qroot);
  
  q = dequeue_quad();
  while(q!=NULL) {
	if(q->elhead!=NULL) {
	  q->flags |= QUAD_BND;
	} else {
	  q->flags |= QUAD_EXT;
	}

	q = dequeue_quad();
  }
}
*/
