/* PUGSAVE.C  Routines to save the triangulation into a file */


#include "pug.h"


/* File format for saved triangulation:

   Nodes: n
   Triangles: t
   x y f
   x y f
   ...
   n1 n2 n3
   n1 n2 n3
   ...

   Where: n is the (integer) number of nodes, t is the (integer) 
   number of triangles, x and y are the coordinates of the
   nodes (doubles), f is an integer flag (not-0 if the node
   is a boundary node, 0 if it isn't), and n1,n2,n3 are the
   (integer) indices of the nodes that form a triangle.

   The boundary nodes are listed in order (clock-wise or
   counter-clockwise) and are always the first group
   of nodes. Internal nodes (non-boundary) are in no
   particular order, nor are the triangles.
*/

int savetriangulation(f,nh,th)
	 FILE *f;	 
	 node *nh;
	 triangle *th;
{
  int i,j;
  node *n;
  triangle *t;
  

  /* First count and assign sequential indices to the nodes.
	 The index field of the structure is used to temporarily 
	 store this index. */
  n = nh;
  i = 0;
  while(n!=NULL) {
	n->index = i++;
	n = n->next;
  }

  t = th;
  j = 0;
  while(t!=NULL) {
	j++;
	t = t->next;
  }

  fprintf(f,"Nodes: %d\nTriangles: %d\n",i,j);

  n = nh;
  while(n!=NULL) {
	fprintf(f,"%g %g %d\n",n->x,n->y,
			((n->flags)&FLAG_RealBoundary) ? FLAG_RealBoundary : 0);
	n = n->next;
  }

  t = th;
  while(t!=NULL) {
	fprintf(f,"%d %d %d\n",t->c[0].n->index,t->c[1].n->index,
			t->c[2].n->index);
	t = t->next;
  }

  return(0);
}


int loadtriangulation(f,nh,th)
	 FILE *f;
	 node **nh;
	 triangle **th;
{
  int i,flags,numnodes,numtriangles,n1,n2,n3;
  node *n,**nodeindex;
  triangle *t;
  double x,y;
  
  if(fscanf(f,"Nodes:%d\nTriangles:%d",&numnodes,&numtriangles)!=2) {
	fprintf(stderr,"Bad input file\n");
	exit(1);
  }

  printf("Loading %d nodes and %d triangles...\n",numnodes,numtriangles);

  nodeindex = (node **)malloc(sizeof(node*)*numnodes);
  if(nodeindex==NULL) {
	fprintf(stderr,"Out of memory. Sorry.\n");
	exit(1);
  }

  for(i=0;i<numnodes;i++) {
	if(fscanf(f,"%lf%lf%d",&x,&y,&flags)!=3) {
	  fprintf(stderr,"Bad input file reading nodes\n");
	  exit(1);
	}
	n = newnode(x,y,flags);
	nodeindex[i] = n;
	if(i>0) (nodeindex[i-1])->next = n;
	else *nh = n;
  }

  for(i=0;i<numtriangles;i++) {
	if(fscanf(f,"%d%d%d",&n1,&n2,&n3)!=3) {
	  fprintf(stderr,"Bad input file reading triangles\n");
	  exit(1);
	}
	make_triangle(th,nodeindex[n1],nodeindex[n2],nodeindex[n3]);
  }

  free(nodeindex);

  printf("Finished loading\n");

  return(0);  
}
