/* PUGQUAD.C  Quadtree based routines */


#include "pug.h"


extern int numclass0;
extern int numclass1;
extern int numclass1adj;
extern int numclass1opp;
extern int numclass2;



/* Remove this definition if input edges are not to be factored
   into the decision to keep on recursing or not */
#define CONSIDER_EDGES

static int quads = 0;

static quadtree *bfsq_first,*bfsq_last;

/* BFS Queue procedures */
static void enqueue_quad(/* quadtree *q; */);
static quadtree *dequeue_quad();

#define empty_queue() (bfsq_first==NULL)
#define in_queue(Q) (((Q)->flags)&QUAD_INQ)


static quadtree *newquad(/* node *nul,*nur,*nbr,*nbl; quadtree *parent; 
					 nodelist *nlhead; edgelist *elhead; */);
static int guaranteecorners(/* nodelist **nlhead; node **nul,**nur,**nbr,**nbl;
						exactdouble sx,sy,ex,ey; */);
static quadtree *build_quadtree(/* exactdouble sx,sy,ex,ey; nodelist *nlhead;
								  edgelist *elhead; */);
static int needtodeepen(/* quadtree *q; */);
void deepenquad(/* quadtree *qstart; */);
static void triangulate_quadtree(/* triangle **th; quadtree *qhead; */);
static node *commonnode(/* edgelist *e1,*e2; */);


node *findnode(/* node *nh; double x,y; */);
static void scalegrid(/* node *nh; double *dim,*sx,*sy; */);


/* Calculates a bounding square that exactly contains the given
   input vertices, and returns the dimensions of the bounding box,
   as well as the coordinates of its lower left corner */
static void scalegrid(nh,dim,sx,sy)
	 node *nh;
	 double *dim,*sx,*sy;
{
  node *n;
  double minx,maxx,miny,maxy;

  if(!scale) {
	if(dim!=NULL) *dim = 1.0;
	if(sx!=NULL) *sx = 0.0;
	if(sy!=NULL) *sy = 0.0;
	return;
  }

  maxx = maxy = 0.0;
  minx = miny = 1e32;

  n = nh;
  while(n!=NULL) {

	if(n->x>maxx) maxx = n->x;
	if(n->x<minx) minx = n->x;
	if(n->y>maxy) maxy = n->x;
	if(n->y<miny) miny = n->x;
	
	n = n->next;
  }

  if(maxx-minx<=0.0 && maxy-miny<=0.0) {
	fprintf(stderr,"Grid has maximum dimension 0.0 or smaller.\nCannot continue\n");
	exit(1);
  }

  if(dim!=NULL) *dim = (maxx-minx > maxy-miny) ? maxx-minx : maxy-miny;

  if(sx!=NULL) *sx = minx;
  if(sy!=NULL) *sy = miny;
}


/* ---------------------------------------------------------------*/

static void enqueue_quad(q)
	 quadtree *q;
{  
  if(bfsq_first==NULL) bfsq_first = q;
  else bfsq_last->next = q;

  bfsq_last = q;

  q->flags |= QUAD_INQ;

  q->next = NULL;  
}

static quadtree *dequeue_quad() {
  quadtree *q;

  q = bfsq_first;
  if(bfsq_first!=NULL) bfsq_first = bfsq_first->next;
  if(bfsq_first==NULL) bfsq_last = NULL;

  if(q!=NULL) q->flags &= ~QUAD_INQ;

  return(q);
}

#define empty_queue() (bfsq_first==NULL)
#define in_queue(Q) (((Q)->flags)&QUAD_INQ)

/* ---------------------------------------------------------------*/


/* Allocates a new quad for the quadtree with default values for
   nodes */
static quadtree *newquad(nul,nur,nbr,nbl,parent,nlhead,elhead) 
	 node *nul,*nur,*nbr,*nbl;
	 quadtree *parent;
	 nodelist *nlhead;
	 edgelist *elhead;
{
  static int maxd;
  quadtree *q;

  q = (quadtree *)malloc(sizeof(quadtree));

  if(q==NULL) {
	fprintf(stderr,"Out of memory building quadtree. Sorry.\n");
	exit(1);
  }

  q->flags = 0;
  q->nul = nul; q->nur = nur; q->nbr = nbr; q->nbl = nbl;
  q->ul = q->ur = q->br = q->bl = NULL;
  q->left = q->right = q->top = q->bottom = NULL;
  q->parent = parent;
  q->next = NULL;

  q->nlhead = nlhead;
  q->elhead = elhead;

  if(parent==NULL) q->depth = 0;
  else q->depth = parent->depth + 1;

  if(q->depth>maxd) {
	maxd = q->depth;
	printf("Quadtree maxdepth now %d (%d nodes)\n",maxd,quads);
  }

  quads++;

  return(q);
}



int listsize(elist)
	 edgelist *elist;
{
  int c;
  edgelist *e;

  e = elist;
  c = 0;
  while(e!=NULL && c<2000) {
	c++;
	e = e->next;
  }

  if(c>=2000) {
	fprintf(stderr,"Error: too many edges in a quad.\n");
	exit(1);
  }

  return(c);
}


static node *localnodes;

/* Allocates a new node and does bookkeeping to keep track
   of nodes */
node *newlocalnode(x,y,flags)
	 double x,y;
	 int flags;
{
  node *n;

  n = newnode(x,y,flags);
  n->prev = NULL;
  n->next = localnodes;
  if(localnodes!=NULL) localnodes->prev = n;
  localnodes = n;

  return(n);
}


/* Extracts the corner nodes from the given list of nodes.
   Returns a value indicating which nodes had to be created
   because they didn't exist. If the return value bit-ored with
   any of the following defines is true then the corresponding
   corner was created: CORNER_UL, CORNER_UR, CORNER_BL, CORNER_BR */
static int guaranteecorners(nlhead,nul,nur,nbr,nbl,sx,sy,ex,ey)
	 nodelist **nlhead;
	 node **nul,**nur,**nbr,**nbl;
	 exactdouble sx,sy,ex,ey;
{
  int corns;

  *nul = *nur = *nbr = *nbr = NULL;
  corns = 0;

  *nbl = extractnode(nlhead,sx,sy);
  *nul = extractnode(nlhead,sx,ey);
  *nbr = extractnode(nlhead,ex,sy);
  *nur = extractnode(nlhead,ex,ey);

  corns = 0;

  if(*nul==NULL) {
	*nul = newlocalnode(ExactDouble2DoubleX(sx),ExactDouble2DoubleY(ey),FLAG_Temp);
	(*nul)->ex = sx; (*nul)->ey = ey;
	corns |= CORNER_UL;
  }

  if(*nur==NULL) {
	*nur = newlocalnode(ExactDouble2DoubleX(ex),ExactDouble2DoubleY(ey),FLAG_Temp);
	(*nur)->ex = ex; (*nur)->ey = ey;
	corns |= CORNER_UR;
  }
  
  if(*nbr==NULL) {
	*nbr = newlocalnode(ExactDouble2DoubleX(ex),ExactDouble2DoubleY(sy),FLAG_Temp);
	(*nbr)->ex = ex; (*nbr)->ey = sy;
	corns |= CORNER_BR;
  }

  if(*nbl==NULL) {
	*nbl = newlocalnode(ExactDouble2DoubleX(sx),ExactDouble2DoubleY(sy),FLAG_Temp);
	(*nbl)->ex = sx; (*nbl)->ey = sy;
	corns |= CORNER_BL;
  }
  
  return(corns);
}
	 

/* Performs a breadth-first walk through the tree creating nodes as
   needed until all conditions are satisfied */
static quadtree *build_quadtree(sx,sy,ex,ey,nlhead,elhead) 
	 exactdouble sx,sy,ex,ey;
	 nodelist *nlhead;
	 edgelist *elhead;
{
  quadtree *qroot;
  node *nul,*nur,*nbl,*nbr;

  printf("Guaranteeing corners...\n");
  /* Ensure that the outermost corners exist */
  guaranteecorners(&nlhead,&nul,&nur,&nbr,&nbl,sx,sy,ex,ey);

  quads = 0;
  printf("Allocating quad..\n");
  qroot = newquad(nul,nur,nbr,nbl,NULL,nlhead,elhead);
  qroot->size = listsize(qroot->elhead);
  qroot->flags |= QUAD_BND | QUAD_EDG;

  printf("Deepening from root...\n");
  deepenquad(qroot);

  return(qroot);
}


void showquad(q)
	 quadtree *q;
{
  printf("Quad: (%g,%g) (%g,%g)\n",
		 q->nbl->x,q->nbl->y,q->nur->x,q->nur->y);
}


/* This routine enqueues the given quadtree and then, as long
   as the queue is not empty it performs a breadth-first
   traversal to deepen the quad as much as necessary, but
   at least one level.
   Each time a node is pulled from the queue it is deepened by
   one. If this causes the neighbours to need deepening, or
   if the children still do then they are enqueued themselves.
   Each leaf quad retains a list of nodes contained within it -- 
   but not including the corners */
void deepenquad(qstart)
	 quadtree *qstart;
{
  int i;
  nodelist *nlul,*nlur,*nlbr,*nlbl;
  edgelist *elul,*elur,*elbr,*elbl;
  node *nl,*nu,*nr,*nb,*nm,*n;
  exactdouble sx,sy,ex,ey,mx,my;
  quadtree *q,*qp;  

  q = qstart;
  while(q!=NULL) {

	if(!IsLeaf(q)) {
	  fprintf(stderr,"Warning: Cannot deepen a non-leaf quad.\n");
	  return;
	}

	if((q->flags)&QUAD_ACU) {
	  printf("Not deepening acute case...\n");
	  q = dequeue_quad();
	  continue;
	}

	sx = q->nbl->ex;
	sy = q->nbl->ey;
	ex = q->nur->ex;
	ey = q->nur->ey;

	mx = (ex-sx)/2 + sx;
	my = (ey-sy)/2 + sy;	

	/* We need to divy up the edgelist among the child quads */
	divyedgelist(q,&elul,&elur,&elbr,&elbl,sx,sy,ex,ey,mx,my);
	q->elhead = NULL;

	/* Remove center and corner nodes from q's nodelist if they exists */
	nm = extractnode(&(q->nlhead),mx,my);
	nu = extractnode(&(q->nlhead),mx,ey);
	nr = extractnode(&(q->nlhead),ex,my);
	nb = extractnode(&(q->nlhead),mx,sy);
	nl = extractnode(&(q->nlhead),sx,my);

	/* This quad is being deepened. The list of nodes living within
	   a quad is only maintained at the base of the quadtree, so
	   the nlhead of the node being deepened must be divided up
	   among its soon-to-exist children, and then destroyed */
	nlul = nlur = nlbr = nlbl = NULL;
	divynodelist(q->nlhead,&nlul,&nlur,&nlbr,&nlbl,sx,sy,ex,ey,mx,my);
	freenodelist(q->nlhead);
	q->nlhead = NULL;

	/* In order to create the children, we must also be sure
	   that the appropriate corners exist for each child. 
	   If we have not already found them in the quad's list,
	   then either one of the siblings created them or they
	   need to be created */

	/* Center-point won't exist in anyone else's list, so if it
	   wasn't there at the beginning then it'll have to be created
	   now */

	if(nm==NULL) {
	  nm = newlocalnode(ExactDouble2DoubleX(mx),ExactDouble2DoubleY(my),FLAG_Temp);
	  nm->ex = mx; nm->ey = my;
	}

	/* For the midpoints of the sides, if they weren't found in the
	   current quad's list, then we have to check the siblings. If
	   the sibling is at least one level deeper, then, there's already
	   a corner with the desired node placement */
	if(nu==NULL) {
	  if(q->top!=NULL && !IsLeaf(q->top)) 
		nu = q->top->bl->nbr;
	  else {
		nu = newlocalnode(ExactDouble2DoubleX(mx),ExactDouble2DoubleY(ey),FLAG_Temp);
		nu->ex = mx; nu->ey = ey;
	  }
	}
	if(nr==NULL) {
	  if(q->right!=NULL && !IsLeaf(q->right)) 
		nr = q->right->bl->nul;
	  else {
		nr = newlocalnode(ExactDouble2DoubleX(ex),ExactDouble2DoubleY(my),FLAG_Temp);
		nr->ex = ex; nr->ey = my;
	  }
	}
	if(nb==NULL) {
	  if(q->bottom!=NULL && !IsLeaf(q->bottom)) 
		nb = q->bottom->ul->nur;
	  else {
		nb = newlocalnode(ExactDouble2DoubleX(mx),ExactDouble2DoubleY(sy),FLAG_Temp);
		nb->ex = mx; nb->ey = sy;
	  }
	}
	if(nl==NULL) {
	  if(q->left!=NULL && !IsLeaf(q->left)) 
		nl = q->left->br->nur;
	  else {
		nl = newlocalnode(ExactDouble2DoubleX(sx),ExactDouble2DoubleY(my),FLAG_Temp);
		nl->ex = sx; nl->ey = my;
	  }
	}

	/* Now, all four corners exist for each node and the node list 
	   has also been properly divided. The children are now created. */
	q->ul = newquad(q->nul,nu,nm,nl,q,nlul,elul); 
	q->ur = newquad(nu,q->nur,nr,nm,q,nlur,elur);
	q->br = newquad(nm,nr,q->nbr,nb,q,nlbr,elbr);
	q->bl = newquad(nl,nm,nb,q->nbl,q,nlbl,elbl);

	if((q->flags)&QUAD_EDG) {
	  if((q->flags)&QUAD_TOP) {q->ul->flags |= QUAD_TOP; q->ur->flags |= QUAD_TOP;}
	  if((q->flags)&QUAD_BOT) {q->bl->flags |= QUAD_BOT; q->br->flags |= QUAD_BOT;}
	  if((q->flags)&QUAD_LEF) {q->bl->flags |= QUAD_LEF; q->ul->flags |= QUAD_LEF;}
	  if((q->flags)&QUAD_RIG) {q->br->flags |= QUAD_RIG; q->ur->flags |= QUAD_RIG;}
	}

	/* Determine whether child quads contains a section of the boundary,
	   or are entirely inside/outside the domain */
	if((q->flags)&QUAD_INT) {

	  /* Parent entirely inside, so children are */
	  q->ul->flags |= QUAD_INT;
	  q->ur->flags |= QUAD_INT;
	  q->br->flags |= QUAD_INT;
	  q->bl->flags |= QUAD_INT;

	} else if((q->flags)&QUAD_EXT) {

	  /* Parent entirely outside, so children are */
	  q->ul->flags |= QUAD_EXT;
	  q->ur->flags |= QUAD_EXT;
	  q->br->flags |= QUAD_EXT;
	  q->bl->flags |= QUAD_EXT;

	} else if((q->flags)&QUAD_BND) {
	  int cnt;

	  /* Parent contains the boundary. Status of each child
		 must be checked */

	  cnt = 0;
	  if(elul!=NULL) {q->ul->flags |= QUAD_BND; cnt++;}
	  if(elur!=NULL) {q->ur->flags |= QUAD_BND; cnt++;}
	  if(elbr!=NULL) {q->br->flags |= QUAD_BND; cnt++;}
	  if(elbl!=NULL) {q->bl->flags |= QUAD_BND; cnt++;}
	  
	  /*if(cnt!=4) 
		traceboundary(q);*/
	}
	q->ul->size = listsize(q->ul->elhead);
	q->ur->size = listsize(q->ur->elhead);
	q->br->size = listsize(q->br->elhead);
	q->bl->size = listsize(q->bl->elhead);

	if(q->ul->size > q->size)
	  fprintf(stderr,"Warning: Parent has %d edges, UL child has %d\n",q->size,q->ul->size);
	else if(q->ur->size > q->size)
	  fprintf(stderr,"Warning: Parent has %d edges, UR child has %d\n",q->size,q->ur->size);
	else if(q->br->size > q->size)
	  fprintf(stderr,"Warning: Parent has %d edges, BR child has %d\n",q->size,q->br->size);
	else if(q->bl->size > q->size)
	  fprintf(stderr,"Warning: Parent has %d edges, BL child has %d\n",q->size,q->bl->size);

	/* Set pointers for sibling relationships */
	q->ul->right = q->ur; q->ul->bottom = q->bl;
	q->ul->top = (q->top==NULL) ? NULL : q->top->bl;
	if(q->ul->top!=NULL) q->ul->top->bottom = q->ul;
	q->ul->left = (q->left==NULL) ? NULL : q->left->ur;
	if(q->ul->left!=NULL) q->ul->left->right = q->ul;

	q->ur->left = q->ul; q->ur->bottom = q->br;
	q->ur->top = (q->top==NULL) ? NULL : q->top->br;
	if(q->ur->top!=NULL) q->ur->top->bottom = q->ur;
	q->ur->right = (q->right==NULL) ? NULL : q->right->ul;
	if(q->ur->right!=NULL) q->ur->right->left = q->ur;

	q->bl->right = q->br; q->bl->top = q->ul;
	q->bl->bottom = (q->bottom==NULL) ? NULL : q->bottom->ul;
	if(q->bl->bottom!=NULL) q->bl->bottom->top = q->bl;
	q->bl->left = (q->left==NULL) ? NULL : q->left->br;
	if(q->bl->left!=NULL) q->bl->left->right = q->bl;

	q->br->left = q->bl; q->br->top = q->ur;
	q->br->bottom = (q->bottom==NULL) ? NULL : q->bottom->ur;
	if(q->br->bottom!=NULL) q->br->bottom->top = q->br;
	q->br->right = (q->right==NULL) ? NULL : q->right->bl;
	if(q->br->right!=NULL) q->br->right->left = q->br;

	/* Deepening one quad can cause the siblings to require deepening
	   as well. The child nodes just created might also be in
	   violation of some other property. Each child and all the
	   siblings if they're one level higher must therefore be
	   evaluated to see if they need to be deepened themselves */

	qp = q->parent;

	if(qp!=NULL) {

	  if(q->left==NULL && qp->left!=NULL && !in_queue(qp->left) &&
		 needtodeepen(qp->left))
		enqueue_quad(qp->left);
	  if(q->top==NULL && qp->top!=NULL && !in_queue(qp->top) &&
		 needtodeepen(qp->top))
		enqueue_quad(qp->top);
	  if(q->right==NULL && qp->right!=NULL && !in_queue(qp->right) &&
		 needtodeepen(qp->right))
		enqueue_quad(qp->right);
	  if(q->bottom==NULL && qp->bottom!=NULL && !in_queue(qp->bottom) &&
		 needtodeepen(qp->bottom))
		enqueue_quad(qp->bottom);
	}

	if(needtodeepen(q->ul)) 
	  enqueue_quad(q->ul);
	if(needtodeepen(q->ur)) 
	  enqueue_quad(q->ur);
	if(needtodeepen(q->br)) 
	  enqueue_quad(q->br);
	if(needtodeepen(q->bl)) 
	  enqueue_quad(q->bl);

	q = dequeue_quad();
  }
}


/* Answers True if the two edges, which are assumed to be adjacent,
   form an acute angle interior to the desired region */
static int acuteinnerangle(e1,e2)
	 edgelist *e1,*e2;
{

  if(e1->n2!=e2->n1 && e1->n1!=e2->n2) {
	fprintf(stderr,"So much for that theory...\n");
  }

  /* Here it is assumed that the edge (n1,n2) has the interior of
	 the region to its left */
  if(e1->n2==e2->n1) {
	if(sarea(e1->n1,e1->n2,e2->n2)!=Sarea_Left) return(False);
  } else if(e1->n1==e2->n2) {
	if(sarea(e2->n1,e2->n2,e1->n2)!=Sarea_Left) return(False);
  }

  /* Now, we know it's less than Pi, but is it less than Pi/2? */
  
  /* INSERT ACUTE-ANGLE TEST HERE */

  return(False);
}

/* Returns a pointer to the shared node if two edges share a node,
   or NULL if they don't */
static node *commonnode(e1,e2) 
	 edgelist *e1,*e2;
{
  if(e1->n1==e2->n1 || e1->n1==e2->n2) return(e1->n1);
  if(e1->n2==e2->n1 || e1->n2==e2->n2) return(e1->n2);
  return(NULL);
}


/* Answers True if q needs to be deepened. At the moment this
   routine is not complete -- doesn't handle the edge issue */
static int needtodeepen(q)
	 quadtree *q;
{
  nodelist *nl;

  if(q->depth<mindepth) return(True);

  /* A quad needs to be further deepened if it contains any input nodes
	 that are not coincident with its four corners. Since the corners
	 are extracted from the list, it is sufficient to examine the list
	 of nodes and see if any of them are input nodes (rather than
	 nodes created by splitting an edge) */

  nl = q->nlhead;
  while(nl!=NULL) {
	if(!((nl->n->flags)&FLAG_Created)) return(True);
	nl = nl->next;
  }

  if((q->flags)&QUAD_ACU) return(False);

  /* The other possibility is due to the edge condition */

  if(q->elhead!=NULL && q->elhead->next!=NULL) {
	node *n;
	edgelist *el1,*el2;
	
	/* Three edges is right out */
	if(q->elhead->next->next!=NULL)	return(True);

	/* Two edges joined at a corner of the square are ok, but
	   two edges not so are not */

	n = commonnode(q->elhead,q->elhead->next);
	if(n==NULL) return(True);

	if(n!=q->nul && n!=q->nur && n!=q->nbr && n!=q->nbl)
	  return(True);

	/* Finally, if two edges are within this square, and they are
	   joined at a corner, and the inner-angle between them is 
	   acute, then the balance condition is no longer relevant */

	if(acuteinnerangle(q->elhead,q->elhead->next)) {
	  q->flags |= QUAD_ACU;
	  fprintf(stderr,"Acute angle case\n");
	  return(False);
	}
  }


  /* Check the balance condition -- that there cannot be an
	 adjacent neighbour of depth greater than q->depth+1 */
  if(q->left!=NULL && !IsLeaf(q->left) && 
	 (!IsLeaf(q->left->ur) || !IsLeaf(q->left->br))) 
	return(True);
  if(q->top!=NULL && !IsLeaf(q->top) && 
	 (!IsLeaf(q->top->bl) || !IsLeaf(q->top->br)))
	return(True);
  if(q->right!=NULL && !IsLeaf(q->right) &&
	 (!IsLeaf(q->right->bl) || !IsLeaf(q->right->ul)))
	return(True);
  if(q->bottom!=NULL && !IsLeaf(q->bottom) &&
	 (!IsLeaf(q->bottom->ul) || !IsLeaf(q->bottom->ur)))
	return(True);

  return(False);
}


/* Traverses the given quadtree looking for leaves to triangulate */
static void triangulate_quadtree(th,qhead)
	 triangle **th;
	 quadtree *qhead;
{

  if(IsLeaf(qhead)) {
	node *nu,*nb,*nl,*nr;

	if(qhead->parent==NULL) {
	  nu = nb = nl = nr = NULL;
	} else {

	  nu = (qhead->top!=NULL && !IsLeaf(qhead->top)) ? qhead->top->bl->nbr : NULL;
	  nl = (qhead->left!=NULL && !IsLeaf(qhead->left)) ? qhead->left->br->nur : NULL;
	  nr = (qhead->right!=NULL && !IsLeaf(qhead->right)) ? qhead->right->bl->nul : NULL;
	  nb = (qhead->bottom!=NULL && !IsLeaf(qhead->bottom)) ? qhead->bottom->ul->nur : NULL;
	}

	triangulate_quad(th,qhead,nu,nr,nb,nl);

  } else {

	triangulate_quadtree(th,qhead->ul);
	triangulate_quadtree(th,qhead->ur);
	triangulate_quadtree(th,qhead->br);
	triangulate_quadtree(th,qhead->bl);
  } 
}


void triangulate_byquadtree(nh,th)
	 node **nh;
	 triangle **th;
{
  node *n;
  nodelist *nlhead,*nl,*nul,*nur,*nbl,*nbr;
  edgelist *elhead;
  quadtree *qh;
  double d;

  printf("Massaging nodes for grid creation...\n");

  /* Determine placement and size of grid */
  scalegrid(*nh,&griddim,&gridsx,&gridsy);

  /* Then project the points into the grid */
  nlhead = NULL;
  n = *nh;
  while(n!=NULL) {

	/* Calculate exactdouble coords */
	n->ex = Double2ExactDoubleX(n->x);
	n->ey = Double2ExactDoubleY(n->y);

	d=fabs(ExactDouble2DoubleX(n->ex)-n->x);
	if(d>maxshift) maxshift = d;
	d=fabs(ExactDouble2DoubleY(n->ey)-n->y);
	if(d>maxshift) maxshift = d;

	/* And ensure they correspond */
	n->x = ExactDouble2DoubleX(n->ex);
	n->y = ExactDouble2DoubleY(n->ey);

	nl = newnodelist(n);

	nl->next = nlhead;
	nlhead = nl;
	
	n = n->next;
  }

#ifdef CONSIDER_EDGES
  elhead = buildedgelist(*nh);
#else
  elhead = NULL;
#endif

  fprintf(stderr,"Building quadtree\n");
  qh = build_quadtree(Double2ExactDoubleX(gridsx),Double2ExactDoubleY(gridsy),
					  Double2ExactDoubleX(gridsx+griddim),Double2ExactDoubleY(gridsy+griddim),
					  nlhead,elhead);

  /* Now, a pass to triangulate the leaves of the quadtree */

  fprintf(stderr,"Triangulating leaves\n");
  triangulate_quadtree(th,qh);
  
  printf("Number of quads in tree: %d\n",quads);
  printf("Quads by type: [0]: %d, [1]: %d, [2]: %d, Total: %d\n",
		 numclass0,numclass1,numclass2,
		 numclass0+numclass1+numclass2);

  if(numclass1!=0) 
	printf("Class 1 quads: [adj]: %d, [opp]: %d\n",numclass1adj,numclass1opp);

  /* Finally, we purge the triangulation of any triangles outside the
	 domain of interest 
  removeexternaltriangles(th,elhead);

  printf("Triangulation cleaned up...\n"); */

  /* And lose the edgelist 
  destroyedgelist(elhead); */
}


