/* PUGREDO.C  Ensures all RealBoundary edges are part of the triangulation.
   If not, this not-especially-efficient-algorithm adds the midpoints
   of each-such-edge-not-present to the node list and recomputes the
   Delaunay triangulation. Repeats until all RealBoundary edges are present.
   Then deletes all triangles outside the RealBoundary polygon. */


#include "pug.h"



/* Prototypes */

int containsalledges(/*node *nh; */);
node *midnode(/*node *n1,*n2; */);
int isedge(/*node *n1,*n2; */);
triangle *incidenttriangle(/*node *n1,*n2; int direction; */);


/* -------------------------------------------------------------------*/
/* Checks that all edges described by the (ordered) list of nodes nh 
   (the first k nodes with realboundary==True) are contained in the 
   triangulation described by the list of triangles th. If not,
   it adds the midpoints of all edges not so contained to the
   point set and retriangulates. Repeats until all needed edges
   are contained in th */
void forceedges1(nh,th)
	 node *nh;
	 triangle **th;
{

  while(!containsalledges(nh)) {

	printf("Retriangulating...\n");

	deletealltriangles(nh,th);

	delaunay(nh,th);

  } 

  purgetriangles(nh,th);
}



/* -------------------------------------------------------------------*/
/* Scans the first part of the list of nodes thay all have their
   FLAG_RealBoundary bit set. If any edge is not present,
   a new node midway is inserted.
   (It is assumed that the last real boundary node is connected
   to the first).
   Returns True if no new nodes added, or False otherwise */
int containsalledges(nh)
	 node *nh;
{
  int b;
  node *n,*newn;

  b=True;
  n=nh;
  while(n!=NULL && ((n->flags)&FLAG_RealBoundary)) {

	if(n->next==NULL || !((n->next->flags)&FLAG_RealBoundary)) {

	  if(!isedge(n,nh)) {

		newn=midnode(n,nh);
		b=False;
	  }

	  break;

	} else {

	  if(!isedge(n,n->next)) {

		newn=midnode(n,n->next);
		b=False;

		/* This next line needed to skip just added node */
		n=n->next;
	  }
	}

	n=n->next;
  }
  
  return(b);
}


/* -------------------------------------------------------------------*/
/* Scans the first part of the list of nodes thay all have their
   FLAG_RealBoundary bit set. If any edge is not present,
   a new node midway is inserted.
   (It is assumed that the last real boundary node is connected
   to the first). */
void forceedges(nh,th)
	 node *nh;
	 triangle **th;
{
  int b,c;
  node *n,*newn,*nextn;

  do {

	b = False;
	
	n=nh;
	while(n!=NULL && ((n->flags)&FLAG_RealBoundary)) {

	  if(n->next==NULL || !((n->next->flags)&FLAG_RealBoundary)) 
		nextn = nh;
	  else
		nextn = n->next;

	  if(!isedge(n,nextn)) {

		newn=midnode(n,nextn);
		c = marktriangles(*th,newn);
		insertpoint(th,newn,c);
		removemarked(th);
		b=True;

		printf("Added (%g,%g)\n",newn->x,newn->y);
		
		/* This next line needed to skip just added node */
		n=n->next;
	  } 		
	  n=n->next;
	}
  } while(b);
}


/* -------------------------------------------------------------------*/
/* Creates a new RealBoundary node between 2 nodes. Initialized thead to NULL,
   (x,y) to the correct spot, and flags to FLAG_RealBoundary. Also inserts
   the node immediately after n1 (but not necessarily immediately before n2) */
node *midnode(n1,n2)
	 node *n1,*n2;
{
  node *n;

  n=newnode(n1->x + (n2->x - n1->x)/2.0,
			n1->y + (n2->y - n1->y)/2.0,
			FLAG_RealBoundary);

  /* and insert into linked list */
  n->next=n1->next;
  n1->next=n;

  return(n);
}


/* -------------------------------------------------------------------*/
/* Deletes all triangles and links to them */
void deletealltriangles(nh,th)
	 node *nh;
	 triangle **th;
{
  node *n;
  triangle *t;
  
  /* Delete entire list of triangles */
  while(*th!=NULL) {

	t=(*th)->next;
	free(*th);
	*th=t;
  }

  /* And remove any links from nodes to triangles */
  n=nh;
  while(n!=NULL) {

	n->thead=NULL;
	n->ttail=NULL;

	n=n->next;
  }
}


/* -------------------------------------------------------------------*/
/* Returns True if (n1,n2) is an edge in the existing triangulation */
int isedge(n1,n2)
	 node *n1,*n2;
{
  int tcorn;
  triangle *t;

  t=n1->thead;
  while(t!=NULL) {

	tcorn=WhichCorner(t,n1);

	if((t->c[(tcorn+1)%3]).n==n2 ||
	   (t->c[(tcorn+2)%3]).n==n2) 
	  return(True);

	t=t->c[tcorn].next;
  }
  
  return(False);
}


/* -------------------------------------------------------------------*/
/* Given 2 nodes that form an edge, and a direction (either Sarea_Left or
   Sarea_Right), this routine locates the triangle on that side
   of the edge and returns a pointer to it */
triangle *incidenttriangle(n1,n2,direction)
	 node *n1,*n2;
	 int direction;
{
  int tcorn;
  node *n3;
  triangle *t;

  t=n1->thead;
  while(t!=NULL) {

	tcorn=WhichCorner(t,n1);

   	n3=NULL;
	if((t->c[(tcorn+1)%3]).n==n2) n3=(t->c[(tcorn+2)%3]).n;
	if((t->c[(tcorn+2)%3]).n==n2) n3=(t->c[(tcorn+1)%3]).n;

	if(n3!=NULL && sarea(n1,n2,n3)==direction)
	  return(t);

	t=t->c[tcorn].next;
  }

  return(NULL);
}



