/* VLSI.C */
/* Implementation for solving the VLSI algorithm assuming ceiling(2/3 N)+2 */
/* chips are O.K. */

/* Made by Raul Silvera M. - Sept. 16, 1.995 */
/* This program is GPL'd - It can be freely distributed and modified, */
/* provided that this header is maintained. Please read the */
/* GNU Public License for details. */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

double drand48();

static long num_tests;

#define BAD 0
#define GOOD 1

typedef struct
{
  char good;			/* Is this chip good or bad? */
  char last_result;		/* chip's response to the last 3-way test */
  char solved;			/* Is this chip's state known to the alg.? */
  char solution;			/* What the alg. says about the chip.*/
} chip;

/* BAD TEST: simulate the behavior of a bad chip. */
int bad_test(chip *a)
{
  a->last_result=GOOD;		/* Can I do something more interesting? */
  return 0;
}

/* TEST: simulate the three-way test of the chips. Keeps a counter of */
/* the number of tests done so far. */
int test(chip *a,chip *b,chip *c)
{
  num_tests++;
  if (a->good)
    a->last_result=b->good && c->good;
  else
    bad_test(a);
  if (b->good)
    b->last_result=a->good && c->good;
  else
    bad_test(b);
  if (c->good)
    c->last_result=a->good && b->good;
  else
    bad_test(c);
  return 0;
}

/* MAKE_RANDOM_CHIPS: create a new random chips structure, given the */
/* total number of chips, and the number of bad chips */
chip *make_random_chips(long num,long bad)
{
  chip *chips;
  long goods_to_go=num-bad,bads_to_go=bad;
  long i=0;
  double good_ratio=((double)(num-bad))/num;

  if (!(chips=(chip *)malloc(sizeof(chip)*num)))
    {
      printf("Sorry. Not enough memory for %dl chips",num);
      exit(0);
    }
  
  for (i=0;i<num;i++)
    {
      if (goods_to_go==0)
	{
	  chips[i].good=0;
	  bads_to_go--;
	}
      else if (bads_to_go==0)
	{
	  chips[i].good=1;
	  goods_to_go--;
	}
      else
	if (drand48()<good_ratio) /* GOOD CHIP */
	  {
	    chips[i].good=1;
	    goods_to_go--;
	  }
	else
	  {
	    chips[i].good=0;
	    bads_to_go--;
	  }
    }
  return chips;
}

/* MAKE_STRING_CHIPS: create the chips structure from a given */
/* string of the form [gb]+, where g represents a good chip, and */
/* b represents a bad chip. */
chip *make_string_chips(char *string,long *pnum,long *pbad)
{
  long num,bad;
  chip *chips;
  long i;
  num=strlen(string);

    
  if (!(chips=(chip *)malloc(sizeof(chip)*num)))
    {
      printf("Sorry. Not enough memory for %l chips",num);
      exit(0);
    }
  
  for (i=0;i<num;i++)
    {
      if (string[i]=='g')
	chips[i].good=1;
      else
	{
	  chips[i].good=0;
	  bad++;
	}
    }
  (*pnum)=num;
  (*pbad)=bad;
  return chips;
}

/* SOLVE_REST: solve the VLSI problem knowing the solution of a subset */
/* of the chips. Simply find two good chips in the known subset and use */
/* them to solve the rest. */
void solve_rest(long num_chips,chip **chips)
{
  long i,good1,good2;

  for (i=0;i<num_chips;i++)
    if (chips[i]->solved && chips[i]->good)
      {
	good1=i++;
	break;
      }
  for (;i<num_chips;i++)
    if (chips[i]->solved && chips[i]->good)
      {
	good2=i;
	break;
      }
  for (i=0;i<num_chips;i++)
    if (!chips[i]->solved)
      {
	test(chips[good1],chips[good2],chips[i]);
	chips[i]->solved=1;
	chips[i]->solution=chips[good1]->last_result;
      }
}

/* SOLVE_TRIVIALLY: solve the VLSI problem trivially, assuming that  */
/* num_chips<9. Simply find a G-G-G trio, and use them to check all */
/* the others.*/
/* Note: If there are only three chips in the set, this algorithm */
/*       doesn't work, since there is no way to determine the good chips. */
/*       This case is handled carefully in the SOLVE function. */
/* Note2: this algorithm is cubic in the number of chips, but since */
/*        it is used only with num_chips<9, it is really O(1).*/
void solve_trivially(long num_chips,chip **chips)
{
  long i,j,k;

  if (num_chips<3)		/* ALL OF THEM ARE O.K. */
    {
      for (i=0;i<num_chips;i++)
	{
	  chips[i]->solved=1;
	  chips[i]->solution=1;
	}
      return;
    }
  else
    for (i=0;i<num_chips-2;i++)
      for (j=i+1;j<num_chips-1;j++)
	for (k=j+1;k<num_chips;k++)
	  {
	    test(chips[i],chips[j],chips[k]);
	    if (chips[i]->last_result && 
		chips[j]->last_result && 
		chips[k]->last_result)
	      {
	        long l;
		
		chips[i]->solved=1;chips[i]->solution=1;
		chips[j]->solved=1;chips[j]->solution=1;
		chips[k]->solved=1;chips[k]->solution=1;
		
		for (l=0;l<num_chips;l++)
		  if (!chips[l]->solved)
		    {
		      test(chips[i],chips[j],chips[l]);
		      chips[l]->solved=1;
		      chips[l]->solution=chips[i]->last_result;
		    }
		return;
	      }
	  }
}

/* SOLVE: recursive algorithm for solving the VLSI problem.*/ 
/* if num_chips is less-or-equal than three, assume that all of them are  */
/* good. */
void solve(long num_chips,chip **chips)
{
  long i;
  long numYYYs=0;

  if (num_chips<9)
    solve_trivially(num_chips,chips);
  else
    {
      for (i=0;i<num_chips-2;i+=3)
	{
	  test(chips[i],chips[i+1],chips[i+2]);
	  if (chips[i]->last_result && 
	      chips[i+1]->last_result && 
	      chips[i+2]->last_result)
	    numYYYs++;
	}
      if (numYYYs>0)
	{
	  chip **new_chips;
	  long num_new_chips=0;

	  if (numYYYs!=3)	/* GENERAL CASE */
	    {
	      new_chips=(chip **)malloc(sizeof(chip *)*numYYYs);
	      for (i=0;i<num_chips-2;i+=3)
		if (chips[i]->last_result && 
		    chips[i+1]->last_result && 
		    chips[i+2]->last_result)
		  new_chips[num_new_chips++]=chips[i];
	    }
	  else		   /* SPECIFIC CASE: since solve(3,chips) doesn't */
	    {		   /* work, take two chips from each trio. */
	      new_chips=(chip **)malloc(sizeof(chip *)*numYYYs*2);
	      for (i=0;i<num_chips-2;i+=3)
		if (chips[i]->last_result && 
		    chips[i+1]->last_result && 
		    chips[i+2]->last_result)
		  {
		    new_chips[num_new_chips++]=chips[i];
		    new_chips[num_new_chips++]=chips[i+1];
		  }
	    }
	  solve(num_new_chips,new_chips);
	  free(new_chips);
	}
      else			/* SPECIFIC CASE: since all trios  */
	solve(6,chips);		/* have the form B-G-G, use any two of 'em */
      solve_rest(num_chips,chips);
    }
}

/* PRINT_SOLUTION: prints the founded solution of the algorithm, including */
/* the number of tests performed, and the solution found. */
/* If the chip's state was correctly detected by the algorithm, a lower-case */
/* letter (g or b) is printed for the chip. A upper-case letter (G or B) is */
/* printer otherwise. A letter N is printed if the algorithm could not */
/* determine the chip's state.*/
void print_solution(long num,chip *chips)
{
  long i,good=0,bad=0;
  printf("Solution for %ld Chips:\n",num);
  for (i=0;i<num;i++)
    {
      if (chips[i].solved)
	{
	  if (chips[i].good)
	    {
/*	      if (chips[i].solution)
		putchar('g');
	      else
		putchar('G');*/
	      good++;
	    }
	  else
	    {
/*	      if (chips[i].solution)
		putchar('B');
	      else
		putchar('b');*/
	      bad++;
	    }
	}
      else
	putchar('N');
    }
  putchar('\n');
  printf("Number of good chips: %ld\nNumber of bad chips: %ld\n",good,bad);
  printf("Number of tests: %ld \n",num_tests);
}

int main(int argc, char **argv)
{
  long num_chips,bad_chips;
  long i;
  chip* chips;
  chip** new_chips;

  if (argc!=2 && argc!=3 && argc!=1)
    {              /* NO INPUT SPECIFIED => MESSAGE ERROR */
      fprintf(stderr,"Usage: %s <num_chips> [num_bad_chips] |<chip-string>\n",argv[0]);
      exit(0);
    }
  if (argc==1)
    {              /* READ FROM STDIN (NUMBER OF GOOD AND BAD CHIPS) */
      char buf[1024],*ptr,*ptr2;
      
      fread(buf,1024,1,stdin);
      num_chips=strtol(buf,&ptr,10);
      bad_chips=strtol(ptr,&ptr2,10);
      if (ptr2==ptr)
	bad_chips=num_chips-(ceil((double)(2*num_chips)/3)+2);
      if (bad_chips<0) 
	bad_chips=0;
      chips=make_random_chips(num_chips,bad_chips);
    }
  else
    {
                   /* READ ARGUMENTS SPECIFICATIONS (NUMBER OF CHIPS) */
    num_chips=atol(argv[1]);
    if (num_chips==0)
      {
	num_chips=strlen(argv[1]);
	chips=make_string_chips(argv[1],&num_chips,&bad_chips);
	if ( ceil((double)(2*num_chips)/3)+ 2 > num_chips-bad_chips)
	  printf("Warning. Good chips are less than ceil(2/3*N)+2\n");
      }
    else
      {
	if (argc==3)
	  {
	    bad_chips=atol(argv[2]);
	    if ( ceil((double)(2*num_chips)/3)+ 2 > num_chips-bad_chips)
	      printf("Warning. Good chips are less than ceil(2/3*N)+2\n");
	  }
	else
	  {
	    bad_chips=num_chips-(ceil((double)(2*num_chips)/3)+2);
	    if (bad_chips<0) 
	      bad_chips=0;
	  }
	chips=make_random_chips(num_chips,bad_chips);
      }
  }
                /* IDENTIFY GOOD AND BAD CHIPS */
  new_chips=malloc(sizeof(chip *)*num_chips); 
  for (i=0;i<num_chips;i++)
    new_chips[i]=&chips[i];
  solve(num_chips,new_chips);
  print_solution(num_chips,chips);
}







