/* machine.c
 * Translate the Kaffe instruction set to the native one.
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>
 */

/* LJH - added s to the following */
#ifdef LJH_JIT
#define DBG(s) s
#define BDBG(s) s
#define MDBG(s) s
#else
#define DBG(s)
#define BDBG(s)
#define MDBG(s)
#endif

#include "config.h"
#include "config-std.h"
#include "config-mem.h"
#include "gtypes.h"
#include "bytecode.h"
#include "slots.h"
#include "registers.h"
#include "seq.h"
#include "machine.h"
#include "basecode.h"
#include "icode.h"
#include "icode-common.h"
#include "labels.h"
#include "codeinfo.h"
#include "codeproto.h"
#include "checks.h"
#include "access.h"
#include "object.h"
#include "constants.h"
#include "baseClasses.h"
#include "classMethod.h"
#include "access.h"
#include "lookup.h"
#include "exception.h"
#include "ops.h"
#include "gc.h"
#include "flags.h"
#include "errors.h"
#include "md.h"

/*
 * Define information about this engine.
 */
char* engine_name = "Just-in-time";
char* engine_version = KVER;

int code_size;
codes* code;

int stackno;
int maxStack;
int maxLocal;
int maxTemp;
int maxArgs;
int maxPush;
int isStatic;

int tmpslot = 0;
int argcount = 0;		/* Function call argument count */
uint32 pc;
uint32 npc;

/* Unit in which code block is increased when overrun */
#define	ALLOCCODEBLOCKSZ	8192
/* Codeblock redzone - allows for safe overrun when generating instructions */
#define	CODEBLOCKREDZONE	256

static nativecode* codeblock;
static int codeblock_size;

/* Mutex to protect the translator from multiple entry */
static iMux translatorMux;

nativecode* CODEPC;
uintp realcodediff;

struct {
	int time;
} jitStats;

static void	generateCode(methods*);
static void	findBlocks(methods*, uint32, int);

void	endBlock(sequence*);
void	startBlock(sequence*);
jlong	currentTime(void);

/*
 * Translate a method into native code.
 */
void
translate(methods* meth)
{
	int wide;
	int i;

	jint low;
	jint high;
	int idx;
	SlotInfo* tmp;
	SlotInfo* tmp2;
	SlotInfo* mtable;

	bytecode* base;
	int len;
	callInfo cinfo;
	Field *field;
	Class* crinfo;

	int64 tms;
	int64 tme;

	/* Only one in the translator at once. Must check the translation
	 * hasn't been done by someone else once we get it.
	 */
	_lockMutex(&translatorMux);
	if (meth->ncode != 0) {
		_unlockMutex(&translatorMux);
		return;
	}

	if (flag_jit) {
		tms = currentTime();
	}

DBG(	printf("callinfo = 0x%x\n", &cinfo);	)

	/* If this code block is native, then just set it up and return */
	if ((meth->accflags & ACC_NATIVE) != 0) {
		native(meth);
		_unlockMutex(&translatorMux);
		return;
	}

#ifdef LJH_JITDUMP
MDBG(   printf("Translating %s.%s%s (%s) 0x%x\n", meth->class->name->data, meth->name->data, meth->signature->data, isStatic ? "static" : "normal", meth->ncode) ;       )
#endif

	maxLocal = meth->localsz;
	maxStack = meth->stacksz;
	maxArgs = meth->ins;
	if (meth->accflags & ACC_STATIC) {
		isStatic = 1;
	}
	else {
		isStatic = 0;
		maxArgs += 1;
	}

	base = (bytecode*)meth->code;
	len = meth->codelen;

	/* Make sure we have enough code space */
	if (len+1 > code_size) {
		if (code != 0) {
			checked_free(code);
		}
		code = checked_calloc(sizeof(codes), len+1);
		code_size = len+1;
	}

	initSeq();
	initRegisters();
	initSlots(maxLocal + maxStack);
	localinfo = &slotinfo[0];
	stackinfo = &localinfo[maxLocal];
	tempinfo = &stackinfo[maxStack];

#ifdef LJH_JITDUMP
        currPC = 0;  /* put some value for the sequence elements
                        that might be generated before evaluation of
                        bytecode starts */
#endif

	/* Traverse code block to mark branch and join points */
	for (i = 0; i <= len; i++) {
		code[i].block = 0;
		code[i].stackdepth = -1;
		code[i].pc = (uintp)-1;
	}
	SET_STACKVAL(len, maxslot);

	/* Scan the code and determine the basic blocks */

	/* First the main body of code. */
	findBlocks(meth, 0, 0);

	/* Then the exception handlers */
	for (i = 0; i < meth->exception_table_len; i++) {
		findBlocks(meth, meth->exception_table[i].handler_pc, 1);
		SET_EXCEPTIONBLOCK_START(meth, meth->exception_table[i].handler_pc);
	}

	/* Clear various counters */
	maxTemp = 0;
	maxPush = 0;
	npc = 0;
	wide = 0;

	/***************************************/
	/* Next reduce bytecode to native code */
	/***************************************/

	start_basic_block();
	start_function();
	monitor_enter();
	if (IS_BASICBLOCK_START(meth, npc)) {
		end_basic_block();
		start_basic_block();
	}

	for (pc = 0; pc < len; pc = npc) {

#ifdef LJH_JITDUMP
                currPC = pc;   /* store the current PC so seq can get it */
#endif

		/* If this is -1 then this code was unreachable - skip it */
		if (STACKVAL(pc) == -1) {
			npc = pc + 1;
			continue;
		}

		/* Retrieve stack for new instruction and next instruction
		 * pointer.
		 */
		stackno = meth->stacksz - STACKVAL(pc);

		assert(stackno >= 0);
		assert(stackno <= meth->stacksz);

		npc = pc + opcodeInfo[base[pc]].len;

		/* Handle WIDEned instructions */
		if (wide) {
			switch(base[pc]) {
			case ILOAD:
			case FLOAD:
			case ALOAD:
			case LLOAD:
			case DLOAD:
			case ISTORE:
			case FSTORE:
			case ASTORE:
			case LSTORE:
			case DSTORE:
				npc += 1;
				break;
			case IINC:
				npc += 2;
				break;
			default:
				assert("Badly widened instruction" == 0);
				break;
			}
		}

		start_instruction();

		/* Note start of exception handling blocks */
		if (IS_EXCEPTIONBLOCK_START(meth, pc)) {
			start_exception_block();
		}

		switch (base[pc]) {
		default:
			fprintf(stderr, "Unknown bytecode %d\n", base[pc]);
			throwException(VerifyError);
			break;
#include "kaffe.def"
		}

		/* Note maximum number of temp slots used and reset it */
		if (tmpslot > maxTemp) {
			maxTemp = tmpslot;
		}
		tmpslot = 0;

		if (IS_BASICBLOCK_START(meth, npc)) {
			end_basic_block();
			start_basic_block();
		}
	}

	assert(maxTemp < MAXTEMPS);

	/* Generate and link code */
MDBG(   printf("Generating Code ...\n"); )  /* LJH -left it in */
	generateCode(meth);
MDBG(   printf("Establishing Method ...\n"); ) /* LJH - left it in */
	establishMethod(meth);

MDBG(	printf("Translated %s.%s%s (%s) 0x%x\n", meth->class->name->data, meth->name->data, meth->signature->data, isStatic ? "static" : "normal", meth->ncode);	)

	if (flag_jit) {
		tme = currentTime();
		jitStats.time += (tme - tms);
		printf("<JIT: %s.%s%s time %dms (%dms) @ 0x%x>\n",
		       CLASS_CNAME(meth->class),
		       meth->name->data, meth->signature->data,
		       (int)(tme - tms), jitStats.time, meth->ncode);   
	}

	_unlockMutex(&translatorMux);
}

/*
 * Analyse code to work out basic blocks and stack depths.
 */
static
void
findBlocks(methods* meth, uint32 pc, int stk)
{
	uint32 destpc;
	uint32 tabpc;
	uint32 idx;
	uint32 low;
	uint32 high;
	int wideflag;
	bytecode* base;
	callInfo cinfo;
	Field *field;
	int len;

	base = (bytecode*)meth->code;
	wideflag = 0;

	while (STACK_NOT_SET(pc)) {
		assert(stk >= 0 && stk <= meth->stacksz);
		SET_STACKVAL(pc, stk);

		if (opcodeInfo[base[pc]].stack == 9) {
			/* Variable length ones */
			bool isStatic = 1;
			idx = (base[pc+1] << 8)|base[pc+2];

			switch (base[pc]) {
			case INVOKEVIRTUAL:
			case INVOKESPECIAL:
			case INVOKEINTERFACE:
				stk -= 1;
				/* ... fall through ... */
			case INVOKESTATIC:
				getMethodSignatureClass(idx, meth, &cinfo);
				stk -= cinfo.in;
				stk += cinfo.out;
				break;

			case GETFIELD:
				stk -= 1;
				isStatic = 0;
				/* ... fall through ... */
			case GETSTATIC:
				field = getField(idx, isStatic, meth);
				stk += FIELD_WSIZE(field);
				break;

			case PUTFIELD:
				stk -= 1;
				isStatic = 0;
				/* ... fall through ... */
			case PUTSTATIC:
				field = getField(idx, isStatic, meth);
				stk -= FIELD_WSIZE(field);
				break;

			case MULTIANEWARRAY:
				stk -= (uint8)base[pc+3] - 1;
				break;

			default:
				goto error;
			}
		}
		else {
			stk += opcodeInfo[base[pc]].stack;
		}

		switch (opcodeInfo[base[pc]].jmp) {
		case 1:
			switch (opcodeInfo[base[pc]].len) {
			case 3:
				destpc = pc + (int16)
					(base[pc+1] * 0x00000100 +
					 base[pc+2] * 0x00000001);
				SET_BASICBLOCK_START(meth, destpc);
				findBlocks(meth, destpc, stk);
				break;
			case 5:
				destpc = pc + (int32)
					(base[pc+1] * 0x01000000 +
					 base[pc+2] * 0x00010000 +
					 base[pc+3] * 0x00000100 +
					 base[pc+4] * 0x00000001);
				SET_BASICBLOCK_START(meth, destpc);
				findBlocks(meth, destpc, stk);
				break;
			default:
				goto error;
			}
			pc += opcodeInfo[base[pc]].len;
			/* Mark start of basic block */
			SET_BASICBLOCK_START(meth, pc);
			break;

		case 2: /* Returns */
			return;

		case 3: /* Gotos */
			switch (opcodeInfo[base[pc]].len) {
			case 3:
				pc = pc + (int16)
					(base[pc+1] * 0x00000100 +
					 base[pc+2] * 0x00000001);
				break;
			case 5:
				pc = pc + (int32)
					(base[pc+1] * 0x01000000 +
					 base[pc+2] * 0x00010000 +
					 base[pc+3] * 0x00000100 +
					 base[pc+4] * 0x00000001);
				break;
			case 0:
				/* Table and Switch */
				tabpc = pc + 4 - (pc % 4);
				if (base[pc] == LOOKUPSWITCH) {
					idx = (uint32)
					      (base[tabpc+4] * 0x01000000 +
					       base[tabpc+5] * 0x00010000 +
					       base[tabpc+6] * 0x00000100 +
					       base[tabpc+7] * 0x00000001);
					for (; idx > 0; idx--) {
						destpc = pc + (int32)
						(base[tabpc+idx*8+4] * 0x01000000 +
					         base[tabpc+idx*8+5] * 0x00010000 +
						 base[tabpc+idx*8+6] * 0x00000100 +
						 base[tabpc+idx*8+7] * 0x00000001);
						SET_BASICBLOCK_START(meth, destpc);
						findBlocks(meth, destpc, stk);
					}
					pc = pc + (int32)
					(base[tabpc+0] * 0x01000000 +
					 base[tabpc+1] * 0x00010000 +
					 base[tabpc+2] * 0x00000100 +
					 base[tabpc+3] * 0x00000001);
				}
				else {
					assert(base[pc] == TABLESWITCH);
					low = (uint32)
					      (base[tabpc+4] * 0x01000000 +
					       base[tabpc+5] * 0x00010000 +
					       base[tabpc+6] * 0x00000100 +
					       base[tabpc+7] * 0x00000001);
					high = (uint32)
					      (base[tabpc+8] * 0x01000000 +
					       base[tabpc+9] * 0x00010000 +
					       base[tabpc+10] * 0x00000100 +
					       base[tabpc+11] * 0x00000001);
					for (idx = 0; idx < high-low+1; idx++) {
						destpc = pc + (int32)
						(base[tabpc+idx*4+12] * 0x01000000 +
					         base[tabpc+idx*4+13] * 0x00010000 +
						 base[tabpc+idx*4+14] * 0x00000100 +
						 base[tabpc+idx*4+15] * 0x00000001);
						SET_BASICBLOCK_START(meth, destpc);
						findBlocks(meth, destpc, stk);
					}
					pc = pc + (int32)
					(base[tabpc+0] * 0x01000000 +
					 base[tabpc+1] * 0x00010000 +
					 base[tabpc+2] * 0x00000100 +
					 base[tabpc+3] * 0x00000001);
				}
				break;
			}
			/* Mark start of basic block */
			SET_BASICBLOCK_START(meth, pc);
			break;

		case 0:
			len = opcodeInfo[base[pc]].len;
			if (base[pc] == WIDE) {
				wideflag = 1;
			}
			/* Handle WIDEned instructions */
			else if (wideflag) {
				switch(base[pc]) {
				case ILOAD:
				case FLOAD:
				case ALOAD:
				case LLOAD:
				case DLOAD:
				case ISTORE:
				case FSTORE:
				case ASTORE:
				case LSTORE:
				case DSTORE:
					len += 1;
					break;
				case IINC:
					len += 2;
					break;
				default:
					assert("Badly widened instruction" == 0);
					break;
				}
			}
			pc += len;
			break;

		case 4:	/* JSR & JSR_W */
			switch (opcodeInfo[base[pc]].len) {
			case 3:
				destpc = pc + (int16)
					(base[pc+1] * 0x00000100 +
					 base[pc+2] * 0x00000001);
				SET_BASICBLOCK_START(meth, destpc);
				findBlocks(meth, destpc, stk+1);
				break;
			case 5:
				destpc = pc + (int32)
					(base[pc+1] * 0x01000000 +
					 base[pc+2] * 0x00010000 +
					 base[pc+3] * 0x00000100 +
					 base[pc+4] * 0x00000001);
				SET_BASICBLOCK_START(meth, destpc);
				findBlocks(meth, destpc, stk+1);
				break;
			default:
				goto error;
			}
			pc += opcodeInfo[base[pc]].len;
			/* Mark start of basic block */
			SET_BASICBLOCK_START(meth, pc);
			break;
		default:
			goto error;
		}
	}
	if (stk != STACKVAL(pc) && STACKVAL(pc) != maxslot) {
		fprintf(stderr, "Stack depths wrong at %d (code=0x%x)\n", pc, &code);
		goto error;
	}
	return;

	/* If we found an inconsistency then throw an exception */
	error:
	throwException(VerifyError);
}

/*
 * Generate the code.
 */
static
void
generateCode(methods* meth)
{
	sequence* t;
	uint32 clen;
	nativecode* realcode;
	int i;
	jexception* e;

DBG(	printf("codeblock = 0x%x\n", codeblock);	)
DBG(	printf("slotinfo = 0x%x\n", slotinfo);		)

	restart:
	CODEPC = codeblock;
	for (t = firstSeq; t != currSeq; t = t->next) {

		/* If we overrun the codeblock, reallocate and restart */
		if (CODEPC >= &codeblock[codeblock_size]) {
			if (codeblock != 0) {
				checked_free(codeblock);
			}
			codeblock_size += ALLOCCODEBLOCKSZ;
			codeblock = checked_calloc(sizeof(nativecode), codeblock_size + CODEBLOCKREDZONE);
			goto restart;
		}

		/* Generate sequences which are actually referenced */
		if (t->ref > 0) {
			(*(t->func))(t);
		}
	}

	/* Establish any code constants */
	establishCodeConstants();

	/* Okay, put this into malloc'ed memory */
	clen = CODEPC - codeblock;
	realcode = checked_malloc(clen);
	memcpy(realcode, codeblock, clen);

	/* Link it */
	realcodediff = realcode - codeblock;
	linkLabels();

	/* Note where it all is */
	meth->ncode = realcode;
	meth->ncode_end = realcode + clen;

	/* Flush code out of cache */
	FLUSH_DCACHE();

	/* Translate exception table and make it available */
	if (meth->exception_table != 0) {
		for (i = 0; i < meth->exception_table_len; i++) {
			e = &meth->exception_table[i];
			e->start_pc = code[e->start_pc].pc + realcodediff;
			e->end_pc = code[e->end_pc].pc + realcodediff;
			e->handler_pc = code[e->handler_pc].pc + realcodediff;
		}
	}
}

/*
 * Start a new instruction.
 */
void
startInsn(sequence* s)
{
	uintp pc;

	/* Note PC value at start of this instruction */
	pc = const_int(2);
	code[pc].pc = (uintp)CODEPC;
}

#if 0
/*
 * Invalidate a specific slot to avoid writeback.
 */
void
invalSlot(sequence* s)
{
	seq_slot(s, 2)->modified = 0;
}
#endif

/*
 * Mark slot not to be written back.
 */
void
nowritebackSlot(sequence* s)
{
	seq_slot(s, 0)->modified |= rnowriteback;
}

/*
 * Start a new basic block.
 */
void
startBlock(sequence* s)
{
	int i;

	startInsn(s);

	/* Invalidate all slots - don't use clobberRegister which will
	 * flush them - we do not want to do that even if they are dirty.
	 */
	for (i = maxslot - 1; i >= 0; i--) {
		if (slotinfo[i].regno != NOREG) {
			register_invalidate(slotinfo[i].regno);
			slot_invalidate(&slotinfo[i]);
		}
	}
/* LJH - left it in */
BDBG(   printf("        --------------------- start block -------------\n"); )

}

/*
 * Fixup after a function call.
 */
void
fixupFunctionCall(sequence* s)
{
	int i;

	startInsn(s);

	/* Invalidate all slots - don't use clobberRegister which will
	 * flush them - we do not want to do that even if they are dirty.
	 */
	for (i = maxslot - 1; i >= 0; i--) {
		if (slotinfo[i].regno != NOREG && (reginfo[slotinfo[i].regno].flags & Rnosaveoncall) == 0) {
			register_invalidate(slotinfo[i].regno);
			slot_invalidate(&slotinfo[i]);
		}
	}
/* LJH - left it in */
BDBG(   printf("            --------------------- after fixup fn call -----\n");)
}

/*
 * End a basic block.
 */
void
endBlock(sequence* s)
{
	int stkno;
	int i;

	/* Spill locals */
	for (i = 0; i < maxLocal; i++) {
		if ((localinfo[i].modified & rwrite) != 0 && localinfo[i].regno != NOREG) {
			if ((localinfo[i].modified & rnowriteback) == 0) {
				spill(&localinfo[i]);
			}
			else {
				localinfo[i].modified &= ~rnowriteback;
			}
		}
	}

	/* Spill stack */
	stkno = const_int(1);
	for (i = stkno; i < maxStack; i++) {
		if ((stackinfo[i].modified & rwrite) != 0 && stackinfo[i].regno != NOREG) {
			if ((stackinfo[i].modified & rnowriteback) == 0) {
				spill(&stackinfo[i]);
			}
			else {
				stackinfo[i].modified &= ~rnowriteback;
			}
		}
	}

	/* Spill temps currently in use */
	tmpslot = const_int(2);
	for (i = 0; i < tmpslot; i++) {
		if ((tempinfo[i].modified & rwrite) != 0 && tempinfo[i].regno != NOREG) {
			if ((tempinfo[i].modified & rnowriteback) == 0) {
				spill(&tempinfo[i]);
			}
			else {
				tempinfo[i].modified &= ~rnowriteback;
			}
		}
	}
/* LJH - left it in */
BDBG(   printf("        --------------------- end block ---------------\n"); )

}

/*
 * Prepare register state for function call.
 */
void
prepareFunctionCall(sequence* s)
{
	int stkno;
	int i;

/* LJH - left it in */
BDBG(   printf("            --------------------- begin prepare fn call ---\n");)
	/* Spill locals */
	for (i = 0; i < maxLocal; i++) {
		if ((localinfo[i].modified & rwrite) != 0 && localinfo[i].regno != NOREG && (reginfo[localinfo[i].regno].flags & Rnosaveoncall) == 0) {
			spill(&localinfo[i]);
		}
	}

	/* Spill stack */
	stkno = const_int(1);
	for (i = stkno; i < maxStack; i++) {
		if ((stackinfo[i].modified & rwrite) != 0 && stackinfo[i].regno != NOREG && (reginfo[stackinfo[i].regno].flags & Rnosaveoncall) == 0) {
			spill(&stackinfo[i]);
		}
	}

	/* Spill temps currently in use */
	tmpslot = const_int(2);
	for (i = 0; i < tmpslot; i++) {
		if ((tempinfo[i].modified & rwrite) != 0 && tempinfo[i].regno != NOREG && (reginfo[tempinfo[i].regno].flags & Rnosaveoncall) == 0) {
			spill(&tempinfo[i]);
		}
	}
}
