/*
 * soft.c
 * Soft instruction support.
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>, 1996.
 */

#define	MDBG(s)
#define	ADBG(s)
#define	CDBG(s)
#define	IDBG(s)

#if MDBG(1) - 1 == 0
#undef CDBG
#define	CDBG(s) s
#endif

#include "config.h"
#include "config-std.h"
#include "config-math.h"
#include <stdarg.h>
#include "gtypes.h"
#include "bytecode.h"
#include "slots.h"
#include "access.h"
#include "object.h"
#include "constants.h"
#include "classMethod.h"
#include "lookup.h"
#include "errors.h"
#include "exception.h"
#include "locks.h"
#include "soft.h"
#include "external.h"
#include "thread.h"
#include "baseClasses.h"

/*
 * soft_new
 */
void*
soft_new(classes* c)
{
	object* obj;

	obj = alloc_object(c, false);

ADBG(	printf("New object of type %s (%d,%x)\n", c->name->data, c->fsize, obj);
		fflush(stdout);						)

	return (obj);
}

/*
 * soft_newarray
 */
void*
soft_newarray(jint type, jint size)
{
	object* obj;

	if (size < 0) {
		throwException(NegativeArraySizeException);
	}

	obj = alloc_array(size, type);

ADBG(	printf("New object of %d type (%d,%x)\n", type, size, obj);
	fflush(stdout);							)

	return (obj);
}

/*
 * soft_anewarray.
 * Allocate a new array whose element type is c.
 */
void*
soft_anewarray(Class* c, jint size)
{
	Class* arr_class = lookupObjectArrayClass (c);
	if (size < 0) {
		throwException(NegativeArraySizeException);
	}

	return (object*)newObject(size * sizeof(void*),
				  arr_class, size, false);
}

/*
 * soft_multianewarray.
 */
#define	MAXDIMS	16

#if defined(INTERPRETER)
void*
soft_multianewarray(classes* class, jint dims, slots* args)
{
        int array[MAXDIMS];
        object* obj;
        jint arg;
        int i; 
	int* arraydims;

        if (dims < MAXDIMS) {
		arraydims = array;
	}
	else {
		arraydims = checked_calloc(dims+1, sizeof(int));
	}

        /* Extract the dimensions into an array */
        for (i = 0; i < dims; i++) {
		arg = args[i].v.tint;
                if (arg < 0) {
                        throwException(NegativeArraySizeException);
                }  
                arraydims[dims-i-1] = arg;
        }
        arraydims[i] = 0;

        /* Mmm, okay now build the array using the wonders of recursion */
        obj = alloc_multiarray(arraydims, CLASS_CNAME (class));

	if (arraydims != array) {
		checked_free(arraydims);
	}

        /* Return the base object */
	return (obj);
}
#endif

#if defined(TRANSLATOR)
void*
soft_multianewarray(classes* class, jint dims, ...)
{
	va_list ap;
	int array[MAXDIMS];
	int i;
	jint arg;
	object* obj;
	int* arraydims;

        if (dims < MAXDIMS) {
		arraydims = array;
	}
	else {
		arraydims = checked_calloc(dims+1, sizeof(int));
	}

	/* Extract the dimensions into an array */
	va_start(ap, dims);
	for (i = 0; i < dims; i++) {
		arg = va_arg(ap, jint);
		if (arg < 0) {
                        throwException(NegativeArraySizeException);
		}
		arraydims[i] = arg;
	}
	arraydims[i] = 0;
	va_end(ap);

	/* Mmm, okay now build the array using the wonders of recursion */
	obj = alloc_multiarray(arraydims, CLASS_CNAME(class));

	if (arraydims != array) {
		checked_free(arraydims);
	}

	/* Return the base object */
	return (obj);
}
#endif

/*
 * soft_monitorenter.
 */
void
soft_monitorenter(object* o)
{
	lockMutex(o);
}

/*
 * soft_monitorexit.
 */
void
soft_monitorexit(object* o)
{
	unlockMutex(o);
}

/*
 * soft_lookupmethod.
 */
void*
soft_lookupmethod(Object* obj, methods* meth)
{
	classes* cls = OBJECT_CLASS (obj);

	meth = findMethod(cls, meth->name, meth->signature);
	if (meth == 0) {
		throwException(NoSuchMethodError);
	}

#if 0 /* MDBG(1) - 1 == -1*/
	/* Fill in dispatchTable and methodTable cache */
	cls->mtable->m[pair->hash % MAXMETHOD].tag = pair;
	cls->mtable->m[pair->hash % MAXMETHOD].method = meth;
	cls->dtable->m[meth->idx].method = meth;
#endif

	return ((void*)meth);
}

#if defined(TRANSLATOR)
nativecode*
soft_get_method_code (Method* meth)
{
	/* If this class needs initialising then do it now */
	if (meth->class->state != CSTATE_OK) {
		initializeClass (meth->class);
	}

	/*
	 * Generate code on demand.
	 */
	if (meth->ncode == 0) {
		translate(meth);
	}

CDBG(	printf("Calling %s:%s%s @ 0x%x\n",
		meth->class->name->data, meth->name->data,
		meth->signature->data, meth->ncode);
		fflush(stdout);
		fflush(stdout);						)

	return (meth->ncode);
}
#endif /* TRANSLATOR */

/*
 * soft_checkcast.
 */
void
soft_checkcast(classes* c, object* o)
{
        /* Null can be cast to anything */
        if (o == 0) {
                return;
        }

	/* If object is instance of class, return */
	if (soft_instanceof(c, o)) {
		return;
	}

	/* Otherwise throw exception */
        throwException(ClassCastException);
}

/*
 * soft_instanceof.
 */
jint
soft_instanceof(classes* c, object* o)
{
	/* Null object are never instances of anything */
	if (o == 0) {
		return (0);
	}

	/* We seperate the casting to objects and the casting to arrays. */
	if (CLASS_CNAME(c)[0] == '[') {
		return (instanceof_array(c, OBJECT_CLASS(o)));
	}
	else {
		return (instanceof(c, OBJECT_CLASS(o)));
	}
}

jint
instanceof(classes* c, classes* oc)
{
	int i;

	if (oc == c) {
		return (1);
	}

	if (oc == 0) {
		return (0);
	}

	if (instanceof(c, oc->superclass)) {
		return (1);
	}

        for (i = 0; i < oc->interface_len; i++) {
                if (instanceof(c, oc->interfaces[i])) {
                        return (1);
                }
        }

        return (0);
}

jint
instanceof_array(classes* c, classes* oc)
{
	char* cn;
	char* ocn;

	cn = CLASS_CNAME(c);
	ocn = CLASS_CNAME(oc);

	/* Skip as many arrays of arrays as we can.  We stop when we find
	 * a base class in either. */
	while (*cn == '[' && *ocn == '[') {
		cn++;
		ocn++;
	}

	switch (*cn) {
	/* If we are still casting to an array then we have failed already */
	case '[':
		return (0);

	/* Casting to an object of some description. */
	case 'L':
		switch (*ocn) {
		/* The only thing we can cast an array to is java/lang/Object.
		 * Checking this here willl save time.
		 */
		case '[':
			if (strcmp(cn, OBJECTCLASSSIG) == 0) {
				return (1);
			}
			return (0);

		/* Casting one object to another.  We short circuit the test
		 * with a check for java/lang/Object since we can always
		 * cast to that.
		 */
		case 'L':
			if (strcmp(cn, OBJECTCLASSSIG) == 0) {
				return (1);
			}
			c = loadClass(makeUtf8Const(cn+1, strlen(cn)-2), 0);
			oc = loadClass(makeUtf8Const(ocn+1, strlen(ocn)-2), 0);
			return (instanceof(c, oc));

		/* Anything else isn't an object so we fail */
		default:
			return (0);
		}

	/* If a base type then we can only cast correctly if they match
	 * exactly.
	 */
	default:
		if (*cn == *ocn) {
			return (1);
		}
		else {
			return (0);
		}
	}
}

/*
 * soft_athrow.
 */
void
soft_athrow(object* o)
{
	throwExternalException(o);
}

/*
 * soft_badarrayindex.
 */
void
soft_badarrayindex(void)
{
	throwException(ArrayIndexOutOfBoundsException);
}

/*
 * soft_nullpointer.
 */
void
soft_nullpointer(void)
{
	throwException(NullPointerException);
}

/*
 * soft_initialise_class.
 */
void
soft_initialise_class (Class* c)
{
	if (c->state != CSTATE_OK) {
		initializeClass (c);
	}
}

/*
 * soft_dcmpg
 */
jint
soft_dcmpg(jdouble v1, jdouble v2)
{
	jint ret;
	if ((!isinf(v1) && isnan(v1)) || (!isinf(v2) && isnan(v2))) {
		ret = 1;
	}
	else if (v1 > v2) {
		ret = 1;
	}
	else if (v1 == v2) {
		ret = 0;
	}
	else {
		ret = -1;
	}

	return (ret);
}

/*
 * soft_dcmpl
 */
jint
soft_dcmpl(jdouble v1, jdouble v2)
{
        jint ret;
	if ((!isinf(v1) && isnan(v1)) || (!isinf(v2) && isnan(v2))) {
		ret = -1;
	}
        else if (v1 > v2) {
                ret = 1;
        }
        else if (v1 == v2) {
                ret = 0;
        }
        else {
                ret = -1;
        }
	return (ret);
}

/*
 * soft_fcmpg
 */
jint
soft_fcmpg(jfloat v1, jfloat v2)
{
        jint ret;
	if ((!isinf(v1) && isnan(v1)) || (!isinf(v2) && isnan(v2))) {
		ret = 1;
	}
        else if (v1 > v2) {
                ret = 1;
        }
        else if (v1 == v2) {
                ret = 0;
        }
        else {
                ret = -1;
        }
	return (ret);
}

/*
 * soft_fcmpg
 */
jint
soft_fcmpl(jfloat v1, jfloat v2)
{
        jint ret;
	if ((!isinf(v1) && isnan(v1)) || (!isinf(v2) && isnan(v2))) {
		ret = -1;
	}
        else if (v1 > v2) {
                ret = 1;
        }
        else if (v1 == v2) {
                ret = 0;
        }
        else {
                ret = -1;
        }
	return (ret);
}

jlong
soft_lmul(jlong v1, jlong v2)
{
	return (v1 * v2);
}

jlong
soft_ldiv(jlong v1, jlong v2)
{
	return (v1 / v2);
}

jlong
soft_lrem(jlong v1, jlong v2)
{
	return (v1 % v2);
}

jfloat
soft_frem(jfloat v1, jfloat v2)
{
	return (remainderf(v1, v2));
}

jdouble
soft_freml(jdouble v1, jdouble v2)
{
	return (remainder(v1, v2));
}

jlong
soft_lshll(jlong v1, jint v2)
{
	return (v1 << (v2 & 63));
}

jlong
soft_ashrl(jlong v1, jint v2)
{
	return (v1 >> (v2 & 63));
}

jlong
soft_lshrl(jlong v1, jint v2)
{
	return (((uint64)v1) >> (v2 & 63));
}

jint
soft_lcmp(jlong v1, jlong v2)
{
	jlong lcc = v2 - v1;
	if (lcc < 0) {
		return (-1);
	}
	else if (lcc > 0) {
		return (1);
	}
	else {
		return (0);
	}
}

jint
soft_mul(jint v1, jint v2)
{
	return (v1*v2);
}

jint
soft_div(jint v1, jint v2)
{
	return (v1/v2);
}

jint
soft_rem(jint v1, jint v2)
{
	return (v1%v2);
}

jfloat
soft_cvtlf(jlong v)
{
	return ((jfloat)v);
}

jdouble
soft_cvtld(jlong v)
{
	return ((jdouble)v);
}

jlong
soft_cvtfl(jfloat v)
{
	return ((jlong)v);
}

jlong
soft_cvtdl(jdouble v)
{
	return ((jlong)v);
}

#ifdef LJH_JITDUMP
/******************************/
char * soft_fn_name(uintp faddr)
/******************************/
/* return a string with the name of a soft function matching faddr */
{  if      (faddr == (uintp) soft_badarrayindex)     
                                                return("soft_badarrayindex");
   else if (faddr == (uintp) soft_new)          return("soft_new");
   else if (faddr == (uintp) soft_newarray)     return("soft_newarray");
   else if (faddr == (uintp) soft_anewarray)    return("soft_anewarray"); 
   else if (faddr == (uintp) soft_multianewarray) 
                                                return("soft_multianewarray");
   else if (faddr == (uintp) soft_monitorenter) return("soft_monitorenter");
   else if (faddr == (uintp) soft_monitorexit)  return("soft_monitorexit");
   else if (faddr == (uintp) soft_lookupmethod) return("soft_loopupmethod");
#if defined(TRANSLATOR)
   else if (faddr == (uintp) soft_get_method_code)
                                                return("soft_get_method_code");
#endif
    else if (faddr == (uintp) soft_checkcast)   return("soft_checkcast");
    else if (faddr == (uintp) soft_instanceof)  return("soft_instanceof");
    else if (faddr == (uintp) soft_athrow)      return("soft_athrow");
    else if (faddr == (uintp) soft_nullpointer) return("soft_nullpointer");
    else if (faddr == (uintp) soft_initialise_class)
                                                return("soft_initialise_class");
    else if (faddr == (uintp) soft_dcmpg)       return("soft_cdmpg");
    else if (faddr == (uintp) soft_dcmpl)       return("soft_dcmpl");
    else if (faddr == (uintp) soft_fcmpg)       return("soft_fcmpg");
    else if (faddr == (uintp) soft_fcmpl)       return("soft_fcmpl");
    else if (faddr == (uintp) soft_lmul)        return("soft_lmul");
    else if (faddr == (uintp) soft_ldiv)        return("soft_ldiv");
    else if (faddr == (uintp) soft_lrem)        return("soft_lrem");
    else if (faddr == (uintp) soft_frem)        return("soft_frem");
    else if (faddr == (uintp) soft_freml)       return("soft_freml");
    else if (faddr == (uintp) soft_lshll)       return("soft_lshll");
    else if (faddr == (uintp) soft_ashrl)       return("soft_ashrl");
    else if (faddr == (uintp) soft_lshrl)       return("soft_lshrl");
    else if (faddr == (uintp) soft_lcmp)        return("soft_lcmp");
    else if (faddr == (uintp) soft_mul)         return("soft_mul");
    else if (faddr == (uintp) soft_div)         return("soft_div");
    else if (faddr == (uintp) soft_rem)         return("soft_rem");
    else if (faddr == (uintp) soft_cvtlf)       return("soft_cvtlf");
    else if (faddr == (uintp) soft_cvtld)       return("soft_cvtld");
    else if (faddr == (uintp) soft_cvtfl)       return("soft_cvtfl");
    else if (faddr == (uintp) soft_cvtdl)       return("soft_cvtdl");
    else                                        return("soft_UNKNOWN");
}
#endif
